/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v11.actnietfschedule.rev20170306.actnietfschedule.configurationschedules;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.yrtietfyangtypes.rev20130715.yrtietfyangtypes.Xpath10;
import org.onosproject.yang.gen.v11.actnietfschedule.rev20170306.actnietfschedule.Operation;
import org.onosproject.yang.gen.v11.actnietfschedule.rev20170306.actnietfschedule.configurationschedules.target.DataValue;
import org.onosproject.yang.gen.v11.actnietfschedule.rev20170306.actnietfschedule.configurationschedules.target.State;
import org.onosproject.yang.gen.v11.actnietfschedule.rev20170306.actnietfschedule.schedules.Schedules;
import org.onosproject.yang.model.InnerModelObject;
import org.onosproject.yang.model.MultiInstanceObject;

/**
 * Represents the implementation of target.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultTarget extends InnerModelObject 
        implements Target, MultiInstanceObject<TargetKeys> {
    protected Xpath10 object;
    protected Operation operation;
    protected BitSet valueLeafFlags = new BitSet();
    protected DataValue dataValue;
    protected State state;
    protected Schedules schedules;

    @Override
    public Xpath10 object() {
        return object;
    }

    @Override
    public Operation operation() {
        return operation;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public DataValue dataValue() {
        return dataValue;
    }

    @Override
    public State state() {
        return state;
    }

    @Override
    public Schedules schedules() {
        return schedules;
    }

    @Override
    public void object(Xpath10 object) {
        valueLeafFlags.set(LeafIdentifier.OBJECT.getLeafIndex());
        this.object = object;
    }

    @Override
    public void operation(Operation operation) {
        valueLeafFlags.set(LeafIdentifier.OPERATION.getLeafIndex());
        this.operation = operation;
    }

    @Override
    public void dataValue(DataValue dataValue) {
        this.dataValue = dataValue;
    }

    @Override
    public void state(State state) {
        this.state = state;
    }

    @Override
    public void schedules(Schedules schedules) {
        this.schedules = schedules;
    }

    @Override
    public int hashCode() {
        return Objects.hash(object, operation, valueLeafFlags, dataValue, state, schedules);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultTarget) {
            DefaultTarget other = (DefaultTarget) obj;
            return
                Objects.deepEquals(object, other.object) &&
                Objects.deepEquals(operation, other.operation) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(dataValue, other.dataValue) &&
                Objects.deepEquals(state, other.state) &&
                Objects.deepEquals(schedules, other.schedules);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("object=" + object)
            .add("operation=" + operation)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("dataValue=" + dataValue)
            .add("state=" + state)
            .add("schedules=" + schedules)
            .toString();
    }


    /**
     * Creates an instance of defaultTarget.
     */
    public DefaultTarget() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}