/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v11.actnietfte.rev20170310.actnietfte.teoptimizationcriterion.optimizationcriterion;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.actnietftetypes.rev20170310.actnietftetypes.PathTiebreakerType;
import org.onosproject.yang.gen.v1.actnietftetypes.rev20170310.actnietftetypes.TePathNumericalProperty;
import org.onosproject.yang.model.InnerModelObject;
import org.onosproject.yang.model.MultiInstanceObject;

/**
 * Represents the implementation of tiebreaker.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultTiebreaker extends InnerModelObject 
        implements Tiebreaker, MultiInstanceObject<TiebreakerKeys> {
    protected Class<? extends TePathNumericalProperty> metric;
    protected Class<? extends PathTiebreakerType> tiebreakerType;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public Class<? extends TePathNumericalProperty> metric() {
        return metric;
    }

    @Override
    public Class<? extends PathTiebreakerType> tiebreakerType() {
        return tiebreakerType;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void metric(Class<? extends TePathNumericalProperty> metric) {
        valueLeafFlags.set(LeafIdentifier.METRIC.getLeafIndex());
        this.metric = metric;
    }

    @Override
    public void tiebreakerType(Class<? extends PathTiebreakerType> tiebreakerType) {
        valueLeafFlags.set(LeafIdentifier.TIEBREAKERTYPE.getLeafIndex());
        this.tiebreakerType = tiebreakerType;
    }

    @Override
    public int hashCode() {
        return Objects.hash(metric, tiebreakerType, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultTiebreaker) {
            DefaultTiebreaker other = (DefaultTiebreaker) obj;
            return
                Objects.deepEquals(metric, other.metric) &&
                Objects.deepEquals(tiebreakerType, other.tiebreakerType) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("metric=" + metric)
            .add("tiebreakerType=" + tiebreakerType)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultTiebreaker.
     */
    public DefaultTiebreaker() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}