/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.transportconstraints.transportconstraints.unicasttransportconstraints;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.SvcId;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.transportconstraintprofile.ConstraintList;
import org.onosproject.yang.model.InnerModelObject;
import org.onosproject.yang.model.MultiInstanceObject;

/**
 * Represents the implementation of constraint.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultConstraint extends InnerModelObject 
        implements Constraint, MultiInstanceObject<ConstraintKeys> {
    protected SvcId constraintId;
    protected SvcId site1;
    protected SvcId site2;
    protected BitSet valueLeafFlags = new BitSet();
    protected List<ConstraintList> constraintList;

    @Override
    public SvcId constraintId() {
        return constraintId;
    }

    @Override
    public SvcId site1() {
        return site1;
    }

    @Override
    public SvcId site2() {
        return site2;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public List<ConstraintList> constraintList() {
        return constraintList;
    }

    @Override
    public void constraintId(SvcId constraintId) {
        valueLeafFlags.set(LeafIdentifier.CONSTRAINTID.getLeafIndex());
        this.constraintId = constraintId;
    }

    @Override
    public void site1(SvcId site1) {
        valueLeafFlags.set(LeafIdentifier.SITE1.getLeafIndex());
        this.site1 = site1;
    }

    @Override
    public void site2(SvcId site2) {
        valueLeafFlags.set(LeafIdentifier.SITE2.getLeafIndex());
        this.site2 = site2;
    }

    @Override
    public void constraintList(List<ConstraintList> constraintList) {
        this.constraintList = constraintList;
    }

    @Override
    public void addToConstraintList(ConstraintList addTo) {
        if (constraintList == null) {
            constraintList = new ArrayList<>();
        }
        constraintList.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(constraintId, site1, site2, valueLeafFlags, constraintList);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultConstraint) {
            DefaultConstraint other = (DefaultConstraint) obj;
            return
                Objects.deepEquals(constraintId, other.constraintId) &&
                Objects.deepEquals(site1, other.site1) &&
                Objects.deepEquals(site2, other.site2) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(constraintList, other.constraintList);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("constraintId=" + constraintId)
            .add("site1=" + site1)
            .add("site2=" + site2)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("constraintList=" + constraintList)
            .toString();
    }


    /**
     * Creates an instance of defaultConstraint.
     */
    public DefaultConstraint() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}