/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.yrtnetworktopology.rev20151208.yrtnetworktopology.networks.network;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.yrtnetworktopology.rev20151208.yrtnetworktopology.LinkId;
import org.onosproject.yang.gen.v1.yrtnetworktopology.rev20151208.yrtnetworktopology.networks.network.augmentedndnetwork.Link;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of augmentedNdNetwork.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultAugmentedNdNetwork extends InnerModelObject implements AugmentedNdNetwork {
    protected LinkId linkId;
    protected BitSet valueLeafFlags = new BitSet();
    protected List<Link> link;

    @Override
    public LinkId linkId() {
        return linkId;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public List<Link> link() {
        return link;
    }

    @Override
    public void linkId(LinkId linkId) {
        valueLeafFlags.set(LeafIdentifier.LINKID.getLeafIndex());
        this.linkId = linkId;
    }

    @Override
    public void link(List<Link> link) {
        this.link = link;
    }

    @Override
    public void addToLink(Link addTo) {
        if (link == null) {
            link = new ArrayList<>();
        }
        link.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(linkId, valueLeafFlags, link);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultAugmentedNdNetwork) {
            DefaultAugmentedNdNetwork other = (DefaultAugmentedNdNetwork) obj;
            return
                Objects.deepEquals(linkId, other.linkId) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(link, other.link);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("linkId=" + linkId)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("link=" + link)
            .toString();
    }


    /**
     * Creates an instance of defaultAugmentedNdNetwork.
     */
    public DefaultAugmentedNdNetwork() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}