/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.yrtsubscription.yrtsubscription.updatepolicy.updatetrigger.onchange;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.yrtsubscription.yrtsubscription.ChangeType;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of augmentedOnChange.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultAugmentedOnChange extends InnerModelObject implements AugmentedOnChange {
    protected boolean noSynchOnStart;
    protected List<ChangeType> excludedChange;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public boolean noSynchOnStart() {
        return noSynchOnStart;
    }

    @Override
    public List<ChangeType> excludedChange() {
        return excludedChange;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void noSynchOnStart(boolean noSynchOnStart) {
        valueLeafFlags.set(LeafIdentifier.NOSYNCHONSTART.getLeafIndex());
        this.noSynchOnStart = noSynchOnStart;
    }

    @Override
    public void excludedChange(List<ChangeType> excludedChange) {
        valueLeafFlags.set(LeafIdentifier.EXCLUDEDCHANGE.getLeafIndex());
        this.excludedChange = excludedChange;
    }

    @Override
    public void addToExcludedChange(ChangeType addTo) {
        if (excludedChange == null) {
            excludedChange = new ArrayList<>();
        }
        excludedChange.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(noSynchOnStart, excludedChange, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultAugmentedOnChange) {
            DefaultAugmentedOnChange other = (DefaultAugmentedOnChange) obj;
            return
                Objects.deepEquals(noSynchOnStart, other.noSynchOnStart) &&
                Objects.deepEquals(excludedChange, other.excludedChange) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("noSynchOnStart=" + noSynchOnStart)
            .add("excludedChange=" + excludedChange)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultAugmentedOnChange.
     */
    public DefaultAugmentedOnChange() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}