/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v11.actnietfte.rev20170310.actnietfte.tetunnelbandwidthtop.bandwidth.autobandwidth.underflow;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.actnietftetypes.rev20170310.actnietftetypes.Percentage;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of state.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultState extends InnerModelObject implements State {
    protected boolean enabled;
    protected Percentage underflowThreshold;
    protected int triggerEventCount;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public boolean enabled() {
        return enabled;
    }

    @Override
    public Percentage underflowThreshold() {
        return underflowThreshold;
    }

    @Override
    public int triggerEventCount() {
        return triggerEventCount;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void enabled(boolean enabled) {
        valueLeafFlags.set(LeafIdentifier.ENABLED.getLeafIndex());
        this.enabled = enabled;
    }

    @Override
    public void underflowThreshold(Percentage underflowThreshold) {
        valueLeafFlags.set(LeafIdentifier.UNDERFLOWTHRESHOLD.getLeafIndex());
        this.underflowThreshold = underflowThreshold;
    }

    @Override
    public void triggerEventCount(int triggerEventCount) {
        valueLeafFlags.set(LeafIdentifier.TRIGGEREVENTCOUNT.getLeafIndex());
        this.triggerEventCount = triggerEventCount;
    }

    @Override
    public int hashCode() {
        return Objects.hash(enabled, underflowThreshold, triggerEventCount, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultState) {
            DefaultState other = (DefaultState) obj;
            return
                Objects.deepEquals(enabled, other.enabled) &&
                Objects.deepEquals(underflowThreshold, other.underflowThreshold) &&
                Objects.deepEquals(triggerEventCount, other.triggerEventCount) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("enabled=" + enabled)
            .add("underflowThreshold=" + underflowThreshold)
            .add("triggerEventCount=" + triggerEventCount)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultState.
     */
    public DefaultState() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}