/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.actnietftetypes.rev20170310.actnietftetypes.explicitroutehopconfig.type.asnumber;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.actnietftetypes.rev20170310.actnietftetypes.TeHopType;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of asNumberHop.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultAsNumberHop extends InnerModelObject implements AsNumberHop {
    protected byte[] asNumber;
    protected TeHopType hopType;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public byte[] asNumber() {
        return asNumber;
    }

    @Override
    public TeHopType hopType() {
        return hopType;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void asNumber(byte[] asNumber) {
        valueLeafFlags.set(LeafIdentifier.ASNUMBER.getLeafIndex());
        this.asNumber = asNumber;
    }

    @Override
    public void hopType(TeHopType hopType) {
        valueLeafFlags.set(LeafIdentifier.HOPTYPE.getLeafIndex());
        this.hopType = hopType;
    }

    @Override
    public int hashCode() {
        return Objects.hash(asNumber, hopType, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultAsNumberHop) {
            DefaultAsNumberHop other = (DefaultAsNumberHop) obj;
            return
                Objects.deepEquals(asNumber, other.asNumber) &&
                Objects.deepEquals(hopType, other.hopType) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("asNumber=" + asNumber)
            .add("hopType=" + hopType)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultAsNumberHop.
     */
    public DefaultAsNumberHop() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}