/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.food.rev20160624.food;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.food.rev20160624.food.food.Snack;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of food.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultFood extends InnerModelObject implements Food {
    protected String p1;
    protected List<String> p2;
    protected BitSet valueLeafFlags = new BitSet();
    protected Snack snack;

    @Override
    public String p1() {
        return p1;
    }

    @Override
    public List<String> p2() {
        return p2;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public Snack snack() {
        return snack;
    }

    @Override
    public void p1(String p1) {
        valueLeafFlags.set(LeafIdentifier.P1.getLeafIndex());
        this.p1 = p1;
    }

    @Override
    public void p2(List<String> p2) {
        valueLeafFlags.set(LeafIdentifier.P2.getLeafIndex());
        this.p2 = p2;
    }

    @Override
    public void snack(Snack snack) {
        this.snack = snack;
    }

    @Override
    public void addToP2(String addTo) {
        if (p2 == null) {
            p2 = new ArrayList<>();
        }
        p2.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(p1, p2, valueLeafFlags, snack);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultFood) {
            DefaultFood other = (DefaultFood) obj;
            return
                Objects.deepEquals(p1, other.p1) &&
                Objects.deepEquals(p2, other.p2) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(snack, other.snack);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("p1=" + p1)
            .add("p2=" + p2)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("snack=" + snack)
            .toString();
    }


    /**
     * Creates an instance of defaultFood.
     */
    public DefaultFood() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}