/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.transportconstraints.transportconstraints.multicasttransportconstraints;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.SvcId;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.transportconstraintprofile.ConstraintList;
import org.onosproject.yang.model.InnerModelObject;
import org.onosproject.yang.model.MultiInstanceObject;

/**
 * Represents the implementation of constraint.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultConstraint extends InnerModelObject 
        implements Constraint, MultiInstanceObject<ConstraintKeys> {
    protected SvcId constraintId;
    protected SvcId srcSite;
    protected SvcId dstSite;
    protected BitSet valueLeafFlags = new BitSet();
    protected List<ConstraintList> constraintList;

    @Override
    public SvcId constraintId() {
        return constraintId;
    }

    @Override
    public SvcId srcSite() {
        return srcSite;
    }

    @Override
    public SvcId dstSite() {
        return dstSite;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public List<ConstraintList> constraintList() {
        return constraintList;
    }

    @Override
    public void constraintId(SvcId constraintId) {
        valueLeafFlags.set(LeafIdentifier.CONSTRAINTID.getLeafIndex());
        this.constraintId = constraintId;
    }

    @Override
    public void srcSite(SvcId srcSite) {
        valueLeafFlags.set(LeafIdentifier.SRCSITE.getLeafIndex());
        this.srcSite = srcSite;
    }

    @Override
    public void dstSite(SvcId dstSite) {
        valueLeafFlags.set(LeafIdentifier.DSTSITE.getLeafIndex());
        this.dstSite = dstSite;
    }

    @Override
    public void constraintList(List<ConstraintList> constraintList) {
        this.constraintList = constraintList;
    }

    @Override
    public void addToConstraintList(ConstraintList addTo) {
        if (constraintList == null) {
            constraintList = new ArrayList<>();
        }
        constraintList.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(constraintId, srcSite, dstSite, valueLeafFlags, constraintList);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultConstraint) {
            DefaultConstraint other = (DefaultConstraint) obj;
            return
                Objects.deepEquals(constraintId, other.constraintId) &&
                Objects.deepEquals(srcSite, other.srcSite) &&
                Objects.deepEquals(dstSite, other.dstSite) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(constraintList, other.constraintList);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("constraintId=" + constraintId)
            .add("srcSite=" + srcSite)
            .add("dstSite=" + dstSite)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("constraintList=" + constraintList)
            .toString();
    }


    /**
     * Creates an instance of defaultConstraint.
     */
    public DefaultConstraint() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}