/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.yrtietfeventnotifications.rev20161027.yrtietfeventnotifications.filters;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.yrtietfeventnotifications.rev20161027.yrtietfeventnotifications.FilterId;
import org.onosproject.yang.gen.v1.yrtietfeventnotifications.rev20161027.yrtietfeventnotifications.basefilter.FilterType;
import org.onosproject.yang.model.InnerModelObject;
import org.onosproject.yang.model.MultiInstanceObject;

/**
 * Represents the implementation of filter.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultFilter extends InnerModelObject 
        implements Filter, MultiInstanceObject<FilterKeys> {
    protected FilterId filterId;
    protected BitSet valueLeafFlags = new BitSet();
    protected FilterType filterType;

    @Override
    public FilterId filterId() {
        return filterId;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public FilterType filterType() {
        return filterType;
    }

    @Override
    public void filterId(FilterId filterId) {
        valueLeafFlags.set(LeafIdentifier.FILTERID.getLeafIndex());
        this.filterId = filterId;
    }

    @Override
    public void filterType(FilterType filterType) {
        this.filterType = filterType;
    }

    @Override
    public int hashCode() {
        return Objects.hash(filterId, valueLeafFlags, filterType);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultFilter) {
            DefaultFilter other = (DefaultFilter) obj;
            return
                Objects.deepEquals(filterId, other.filterId) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(filterType, other.filterType);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("filterId=" + filterId)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("filterType=" + filterType)
            .toString();
    }


    /**
     * Creates an instance of defaultFilter.
     */
    public DefaultFilter() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}