/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v11.actnietfte.rev20170310.actnietfte.bidirassocproperties.bidirectional;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.actnietftetypes.rev20170310.actnietftetypes.BidirAssociationType;
import org.onosproject.yang.gen.v1.actnietftetypes.rev20170310.actnietftetypes.BidirProvisioningMode;
import org.onosproject.yang.gen.v1.yrtietfinettypes.rev20130715.yrtietfinettypes.IpAddress;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of association.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultAssociation extends InnerModelObject implements Association {
    protected int id;
    protected IpAddress source;
    protected IpAddress globalSource;
    protected Class<? extends BidirAssociationType> type;
    protected Class<? extends BidirProvisioningMode> provisioing;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public int id() {
        return id;
    }

    @Override
    public IpAddress source() {
        return source;
    }

    @Override
    public IpAddress globalSource() {
        return globalSource;
    }

    @Override
    public Class<? extends BidirAssociationType> type() {
        return type;
    }

    @Override
    public Class<? extends BidirProvisioningMode> provisioing() {
        return provisioing;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void id(int id) {
        valueLeafFlags.set(LeafIdentifier.ID.getLeafIndex());
        this.id = id;
    }

    @Override
    public void source(IpAddress source) {
        valueLeafFlags.set(LeafIdentifier.SOURCE.getLeafIndex());
        this.source = source;
    }

    @Override
    public void globalSource(IpAddress globalSource) {
        valueLeafFlags.set(LeafIdentifier.GLOBALSOURCE.getLeafIndex());
        this.globalSource = globalSource;
    }

    @Override
    public void type(Class<? extends BidirAssociationType> type) {
        valueLeafFlags.set(LeafIdentifier.TYPE.getLeafIndex());
        this.type = type;
    }

    @Override
    public void provisioing(Class<? extends BidirProvisioningMode> provisioing) {
        valueLeafFlags.set(LeafIdentifier.PROVISIOING.getLeafIndex());
        this.provisioing = provisioing;
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, source, globalSource, type, provisioing, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultAssociation) {
            DefaultAssociation other = (DefaultAssociation) obj;
            return
                Objects.deepEquals(id, other.id) &&
                Objects.deepEquals(source, other.source) &&
                Objects.deepEquals(globalSource, other.globalSource) &&
                Objects.deepEquals(type, other.type) &&
                Objects.deepEquals(provisioing, other.provisioing) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("id=" + id)
            .add("source=" + source)
            .add("globalSource=" + globalSource)
            .add("type=" + type)
            .add("provisioing=" + provisioing)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultAssociation.
     */
    public DefaultAssociation() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}