/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.actnietfroutingtypes.rev20170227.actnietfroutingtypes.mplslabelstack.mplslabelstack;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.actnietfroutingtypes.rev20170227.actnietfroutingtypes.MplsLabel;
import org.onosproject.yang.model.InnerModelObject;
import org.onosproject.yang.model.MultiInstanceObject;

/**
 * Represents the implementation of entry.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultEntry extends InnerModelObject 
        implements Entry, MultiInstanceObject<EntryKeys> {
    protected short id;
    protected MplsLabel label;
    protected short ttl;
    protected short trafficClass;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public short id() {
        return id;
    }

    @Override
    public MplsLabel label() {
        return label;
    }

    @Override
    public short ttl() {
        return ttl;
    }

    @Override
    public short trafficClass() {
        return trafficClass;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void id(short id) {
        valueLeafFlags.set(LeafIdentifier.ID.getLeafIndex());
        this.id = id;
    }

    @Override
    public void label(MplsLabel label) {
        valueLeafFlags.set(LeafIdentifier.LABEL.getLeafIndex());
        this.label = label;
    }

    @Override
    public void ttl(short ttl) {
        valueLeafFlags.set(LeafIdentifier.TTL.getLeafIndex());
        this.ttl = ttl;
    }

    @Override
    public void trafficClass(short trafficClass) {
        valueLeafFlags.set(LeafIdentifier.TRAFFICCLASS.getLeafIndex());
        this.trafficClass = trafficClass;
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, label, ttl, trafficClass, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultEntry) {
            DefaultEntry other = (DefaultEntry) obj;
            return
                Objects.deepEquals(id, other.id) &&
                Objects.deepEquals(label, other.label) &&
                Objects.deepEquals(ttl, other.ttl) &&
                Objects.deepEquals(trafficClass, other.trafficClass) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("id=" + id)
            .add("label=" + label)
            .add("ttl=" + ttl)
            .add("trafficClass=" + trafficClass)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultEntry.
     */
    public DefaultEntry() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}