/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.siterouting.routingprotocols.routingprotocol;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.AddressFamily;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.siterouting.routingprotocols.routingprotocol.ospf.ShamLinks;
import org.onosproject.yang.gen.v1.yrtietfyangtypes.rev20130715.yrtietfyangtypes.DottedQuad;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of ospf.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultOspf extends InnerModelObject implements Ospf {
    protected DottedQuad areaAddress;
    protected int metric;
    protected List<Class<? extends AddressFamily>> addressFamily;
    protected BitSet valueLeafFlags = new BitSet();
    protected ShamLinks shamLinks;

    @Override
    public DottedQuad areaAddress() {
        return areaAddress;
    }

    @Override
    public int metric() {
        return metric;
    }

    @Override
    public List<Class<? extends AddressFamily>> addressFamily() {
        return addressFamily;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public ShamLinks shamLinks() {
        return shamLinks;
    }

    @Override
    public void areaAddress(DottedQuad areaAddress) {
        valueLeafFlags.set(LeafIdentifier.AREAADDRESS.getLeafIndex());
        this.areaAddress = areaAddress;
    }

    @Override
    public void metric(int metric) {
        valueLeafFlags.set(LeafIdentifier.METRIC.getLeafIndex());
        this.metric = metric;
    }

    @Override
    public void addressFamily(List<Class<? extends AddressFamily>> addressFamily) {
        valueLeafFlags.set(LeafIdentifier.ADDRESSFAMILY.getLeafIndex());
        this.addressFamily = addressFamily;
    }

    @Override
    public void shamLinks(ShamLinks shamLinks) {
        this.shamLinks = shamLinks;
    }

    @Override
    public void addToAddressFamily(Class<? extends AddressFamily> addTo) {
        if (addressFamily == null) {
            addressFamily = new ArrayList<>();
        }
        addressFamily.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(areaAddress, metric, addressFamily, valueLeafFlags, shamLinks);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultOspf) {
            DefaultOspf other = (DefaultOspf) obj;
            return
                Objects.deepEquals(areaAddress, other.areaAddress) &&
                Objects.deepEquals(metric, other.metric) &&
                Objects.deepEquals(addressFamily, other.addressFamily) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(shamLinks, other.shamLinks);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("areaAddress=" + areaAddress)
            .add("metric=" + metric)
            .add("addressFamily=" + addressFamily)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("shamLinks=" + shamLinks)
            .toString();
    }


    /**
     * Creates an instance of defaultOspf.
     */
    public DefaultOspf() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}