/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.ymstopology.rev20140101.ymstopology;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.ymstopology.rev20140101.ymstopology.node.Choice1;
import org.onosproject.yang.gen.v1.ymstopology.rev20140101.ymstopology.node.TerminationPoints;
import org.onosproject.yang.model.InnerModelObject;
import org.onosproject.yang.model.MultiInstanceObject;

/**
 * Represents the implementation of node.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultNode extends InnerModelObject 
        implements Node, MultiInstanceObject<NodeKeys> {
    protected String nodeId;
    protected List<String> nodeProp;
    protected BitSet valueLeafFlags = new BitSet();
    protected TerminationPoints terminationPoints;
    protected Choice1 choice1;

    @Override
    public String nodeId() {
        return nodeId;
    }

    @Override
    public List<String> nodeProp() {
        return nodeProp;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public TerminationPoints terminationPoints() {
        return terminationPoints;
    }

    @Override
    public Choice1 choice1() {
        return choice1;
    }

    @Override
    public void nodeId(String nodeId) {
        valueLeafFlags.set(LeafIdentifier.NODEID.getLeafIndex());
        this.nodeId = nodeId;
    }

    @Override
    public void nodeProp(List<String> nodeProp) {
        valueLeafFlags.set(LeafIdentifier.NODEPROP.getLeafIndex());
        this.nodeProp = nodeProp;
    }

    @Override
    public void terminationPoints(TerminationPoints terminationPoints) {
        this.terminationPoints = terminationPoints;
    }

    @Override
    public void choice1(Choice1 choice1) {
        this.choice1 = choice1;
    }

    @Override
    public void addToNodeProp(String addTo) {
        if (nodeProp == null) {
            nodeProp = new ArrayList<>();
        }
        nodeProp.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(nodeId, nodeProp, valueLeafFlags, terminationPoints, choice1);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultNode) {
            DefaultNode other = (DefaultNode) obj;
            return
                Objects.deepEquals(nodeId, other.nodeId) &&
                Objects.deepEquals(nodeProp, other.nodeProp) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(terminationPoints, other.terminationPoints) &&
                Objects.deepEquals(choice1, other.choice1);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("nodeId=" + nodeId)
            .add("nodeProp=" + nodeProp)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("terminationPoints=" + terminationPoints)
            .add("choice1=" + choice1)
            .toString();
    }


    /**
     * Creates an instance of defaultNode.
     */
    public DefaultNode() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}