/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v11.yrtietfrestconf.rev20170126.yrtietfrestconf.errors.errors;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v11.yrtietfrestconf.rev20170126.yrtietfrestconf.errors.errors.error.ErrorInfo;
import org.onosproject.yang.gen.v11.yrtietfrestconf.rev20170126.yrtietfrestconf.errors.errors.error.ErrorTypeEnum;
import org.onosproject.yang.model.InnerModelObject;
import org.onosproject.yang.model.MultiInstanceObject;

/**
 * Represents the implementation of error.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultError extends InnerModelObject 
        implements Error, MultiInstanceObject<ErrorKeys> {
    protected ErrorTypeEnum errorType;
    protected String errorTag;
    protected String errorAppTag;
    protected String errorPath;
    protected String errorMessage;
    protected BitSet valueLeafFlags = new BitSet();
    protected ErrorInfo errorInfo;

    @Override
    public ErrorTypeEnum errorType() {
        return errorType;
    }

    @Override
    public String errorTag() {
        return errorTag;
    }

    @Override
    public String errorAppTag() {
        return errorAppTag;
    }

    @Override
    public String errorPath() {
        return errorPath;
    }

    @Override
    public String errorMessage() {
        return errorMessage;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public ErrorInfo errorInfo() {
        return errorInfo;
    }

    @Override
    public void errorType(ErrorTypeEnum errorType) {
        valueLeafFlags.set(LeafIdentifier.ERRORTYPE.getLeafIndex());
        this.errorType = errorType;
    }

    @Override
    public void errorTag(String errorTag) {
        valueLeafFlags.set(LeafIdentifier.ERRORTAG.getLeafIndex());
        this.errorTag = errorTag;
    }

    @Override
    public void errorAppTag(String errorAppTag) {
        valueLeafFlags.set(LeafIdentifier.ERRORAPPTAG.getLeafIndex());
        this.errorAppTag = errorAppTag;
    }

    @Override
    public void errorPath(String errorPath) {
        valueLeafFlags.set(LeafIdentifier.ERRORPATH.getLeafIndex());
        this.errorPath = errorPath;
    }

    @Override
    public void errorMessage(String errorMessage) {
        valueLeafFlags.set(LeafIdentifier.ERRORMESSAGE.getLeafIndex());
        this.errorMessage = errorMessage;
    }

    @Override
    public void errorInfo(ErrorInfo errorInfo) {
        this.errorInfo = errorInfo;
    }

    @Override
    public int hashCode() {
        return Objects.hash(errorType, errorTag, errorAppTag, errorPath, errorMessage, valueLeafFlags, errorInfo);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultError) {
            DefaultError other = (DefaultError) obj;
            return
                Objects.deepEquals(errorType, other.errorType) &&
                Objects.deepEquals(errorTag, other.errorTag) &&
                Objects.deepEquals(errorAppTag, other.errorAppTag) &&
                Objects.deepEquals(errorPath, other.errorPath) &&
                Objects.deepEquals(errorMessage, other.errorMessage) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(errorInfo, other.errorInfo);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("errorType=" + errorType)
            .add("errorTag=" + errorTag)
            .add("errorAppTag=" + errorAppTag)
            .add("errorPath=" + errorPath)
            .add("errorMessage=" + errorMessage)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("errorInfo=" + errorInfo)
            .toString();
    }


    /**
     * Creates an instance of defaultError.
     */
    public DefaultError() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}