/*
 * Copyright 2017-present Open Networking Laboratory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.yrtietftetopology.rev20160317.yrtietftetopology.telinkconfig.bundlestacklevel.component.componentlinks;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.model.InnerModelObject;
import org.onosproject.yang.model.MultiInstanceObject;

/**
 * Represents the implementation of componentLink.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultComponentLink extends InnerModelObject 
        implements ComponentLink, MultiInstanceObject<ComponentLinkKeys> {
    protected long sequence;
    protected String srcInterfaceRef;
    protected String desInterfaceRef;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public long sequence() {
        return sequence;
    }

    @Override
    public String srcInterfaceRef() {
        return srcInterfaceRef;
    }

    @Override
    public String desInterfaceRef() {
        return desInterfaceRef;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void sequence(long sequence) {
        valueLeafFlags.set(LeafIdentifier.SEQUENCE.getLeafIndex());
        this.sequence = sequence;
    }

    @Override
    public void srcInterfaceRef(String srcInterfaceRef) {
        valueLeafFlags.set(LeafIdentifier.SRCINTERFACEREF.getLeafIndex());
        this.srcInterfaceRef = srcInterfaceRef;
    }

    @Override
    public void desInterfaceRef(String desInterfaceRef) {
        valueLeafFlags.set(LeafIdentifier.DESINTERFACEREF.getLeafIndex());
        this.desInterfaceRef = desInterfaceRef;
    }

    @Override
    public int hashCode() {
        return Objects.hash(sequence, srcInterfaceRef, desInterfaceRef, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultComponentLink) {
            DefaultComponentLink other = (DefaultComponentLink) obj;
            return
                Objects.equals(sequence, other.sequence) &&
                Objects.equals(srcInterfaceRef, other.srcInterfaceRef) &&
                Objects.equals(desInterfaceRef, other.desInterfaceRef) &&
                Objects.equals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("sequence=" + sequence)
            .add("srcInterfaceRef=" + srcInterfaceRef)
            .add("desInterfaceRef=" + desInterfaceRef)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultComponentLink.
     */
    public DefaultComponentLink() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}