/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.logisticsmanager.rev20160524.logisticsmanager;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of purchasingSupervisor.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultPurchasingSupervisor extends InnerModelObject implements PurchasingSupervisor {
    protected String purchasingSpecialist;
    protected List<String> support;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public String purchasingSpecialist() {
        return purchasingSpecialist;
    }

    @Override
    public List<String> support() {
        return support;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void purchasingSpecialist(String purchasingSpecialist) {
        valueLeafFlags.set(LeafIdentifier.PURCHASINGSPECIALIST.getLeafIndex());
        this.purchasingSpecialist = purchasingSpecialist;
    }

    @Override
    public void support(List<String> support) {
        valueLeafFlags.set(LeafIdentifier.SUPPORT.getLeafIndex());
        this.support = support;
    }

    @Override
    public void addToSupport(String addTo) {
        if (support == null) {
            support = new ArrayList<>();
        }
        support.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(purchasingSpecialist, support, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultPurchasingSupervisor) {
            DefaultPurchasingSupervisor other = (DefaultPurchasingSupervisor) obj;
            return
                Objects.equals(purchasingSpecialist, other.purchasingSpecialist) &&
                Objects.equals(support, other.support) &&
                Objects.equals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("purchasingSpecialist=" + purchasingSpecialist)
            .add("support=" + support)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultPurchasingSupervisor.
     */
    public DefaultPurchasingSupervisor() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}