/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.yrtietftetypes.rev20160320.yrtietftetypes.tunnelpathselection;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.yrtietftetypes.rev20160320.yrtietftetypes.PathMetricType;
import org.onosproject.yang.gen.v1.yrtietftetypes.rev20160320.yrtietftetypes.PathTiebreakerType;
import org.onosproject.yang.gen.v1.yrtietftetypes.rev20160320.yrtietftetypes.TopologyId;
import org.onosproject.yang.gen.v1.yrtietftetypes.rev20160320.yrtietftetypes.tunnelpathaffinities.TunnelPathAffinities;
import org.onosproject.yang.gen.v1.yrtietftetypes.rev20160320.yrtietftetypes.tunnelpathsrlgs.TunnelPathSrlgs;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of pathSelection.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultPathSelection extends InnerModelObject implements PathSelection {
    protected TopologyId topology;
    protected long costLimit;
    protected short hopLimit;
    protected Class<? extends PathMetricType> metricType;
    protected Class<? extends PathTiebreakerType> tiebreakerType;
    protected boolean ignoreOverload;
    protected BitSet valueLeafFlags = new BitSet();
    protected TunnelPathAffinities tunnelPathAffinities;
    protected TunnelPathSrlgs tunnelPathSrlgs;

    @Override
    public TopologyId topology() {
        return topology;
    }

    @Override
    public long costLimit() {
        return costLimit;
    }

    @Override
    public short hopLimit() {
        return hopLimit;
    }

    @Override
    public Class<? extends PathMetricType> metricType() {
        return metricType;
    }

    @Override
    public Class<? extends PathTiebreakerType> tiebreakerType() {
        return tiebreakerType;
    }

    @Override
    public boolean ignoreOverload() {
        return ignoreOverload;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public TunnelPathAffinities tunnelPathAffinities() {
        return tunnelPathAffinities;
    }

    @Override
    public TunnelPathSrlgs tunnelPathSrlgs() {
        return tunnelPathSrlgs;
    }

    @Override
    public void topology(TopologyId topology) {
        valueLeafFlags.set(LeafIdentifier.TOPOLOGY.getLeafIndex());
        this.topology = topology;
    }

    @Override
    public void costLimit(long costLimit) {
        valueLeafFlags.set(LeafIdentifier.COSTLIMIT.getLeafIndex());
        this.costLimit = costLimit;
    }

    @Override
    public void hopLimit(short hopLimit) {
        valueLeafFlags.set(LeafIdentifier.HOPLIMIT.getLeafIndex());
        this.hopLimit = hopLimit;
    }

    @Override
    public void metricType(Class<? extends PathMetricType> metricType) {
        valueLeafFlags.set(LeafIdentifier.METRICTYPE.getLeafIndex());
        this.metricType = metricType;
    }

    @Override
    public void tiebreakerType(Class<? extends PathTiebreakerType> tiebreakerType) {
        valueLeafFlags.set(LeafIdentifier.TIEBREAKERTYPE.getLeafIndex());
        this.tiebreakerType = tiebreakerType;
    }

    @Override
    public void ignoreOverload(boolean ignoreOverload) {
        valueLeafFlags.set(LeafIdentifier.IGNOREOVERLOAD.getLeafIndex());
        this.ignoreOverload = ignoreOverload;
    }

    @Override
    public void tunnelPathAffinities(TunnelPathAffinities tunnelPathAffinities) {
        this.tunnelPathAffinities = tunnelPathAffinities;
    }

    @Override
    public void tunnelPathSrlgs(TunnelPathSrlgs tunnelPathSrlgs) {
        this.tunnelPathSrlgs = tunnelPathSrlgs;
    }

    @Override
    public int hashCode() {
        return Objects.hash(topology, costLimit, hopLimit, metricType, tiebreakerType, ignoreOverload, valueLeafFlags, tunnelPathAffinities, tunnelPathSrlgs);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultPathSelection) {
            DefaultPathSelection other = (DefaultPathSelection) obj;
            return
                Objects.deepEquals(topology, other.topology) &&
                Objects.deepEquals(costLimit, other.costLimit) &&
                Objects.deepEquals(hopLimit, other.hopLimit) &&
                Objects.deepEquals(metricType, other.metricType) &&
                Objects.deepEquals(tiebreakerType, other.tiebreakerType) &&
                Objects.deepEquals(ignoreOverload, other.ignoreOverload) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(tunnelPathAffinities, other.tunnelPathAffinities) &&
                Objects.deepEquals(tunnelPathSrlgs, other.tunnelPathSrlgs);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("topology=" + topology)
            .add("costLimit=" + costLimit)
            .add("hopLimit=" + hopLimit)
            .add("metricType=" + metricType)
            .add("tiebreakerType=" + tiebreakerType)
            .add("ignoreOverload=" + ignoreOverload)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("tunnelPathAffinities=" + tunnelPathAffinities)
            .add("tunnelPathSrlgs=" + tunnelPathSrlgs)
            .toString();
    }


    /**
     * Creates an instance of defaultPathSelection.
     */
    public DefaultPathSelection() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}