/*
 * Copyright 2019-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.yrtietfschedule.rev20160301.yrtietfschedule.schedules.schedules;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.yrtietfyangtypes.rev20130715.yrtietfyangtypes.DateAndTime;
import org.onosproject.yang.model.InnerModelObject;
import org.onosproject.yang.model.MultiInstanceObject;

/**
 * Represents the implementation of schedule.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultSchedule extends InnerModelObject 
        implements Schedule, MultiInstanceObject<ScheduleKeys> {
    protected long scheduleId;
    protected DateAndTime start;
    protected String scheduleDuration;
    protected String repeatInterval;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public long scheduleId() {
        return scheduleId;
    }

    @Override
    public DateAndTime start() {
        return start;
    }

    @Override
    public String scheduleDuration() {
        return scheduleDuration;
    }

    @Override
    public String repeatInterval() {
        return repeatInterval;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void scheduleId(long scheduleId) {
        valueLeafFlags.set(LeafIdentifier.SCHEDULEID.getLeafIndex());
        this.scheduleId = scheduleId;
    }

    @Override
    public void start(DateAndTime start) {
        valueLeafFlags.set(LeafIdentifier.START.getLeafIndex());
        this.start = start;
    }

    @Override
    public void scheduleDuration(String scheduleDuration) {
        valueLeafFlags.set(LeafIdentifier.SCHEDULEDURATION.getLeafIndex());
        this.scheduleDuration = scheduleDuration;
    }

    @Override
    public void repeatInterval(String repeatInterval) {
        valueLeafFlags.set(LeafIdentifier.REPEATINTERVAL.getLeafIndex());
        this.repeatInterval = repeatInterval;
    }

    @Override
    public int hashCode() {
        return Objects.hash(scheduleId, start, scheduleDuration, repeatInterval, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultSchedule) {
            DefaultSchedule other = (DefaultSchedule) obj;
            return
                Objects.deepEquals(scheduleId, other.scheduleId) &&
                Objects.deepEquals(start, other.start) &&
                Objects.deepEquals(scheduleDuration, other.scheduleDuration) &&
                Objects.deepEquals(repeatInterval, other.repeatInterval) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("scheduleId=" + scheduleId)
            .add("start=" + start)
            .add("scheduleDuration=" + scheduleDuration)
            .add("repeatInterval=" + repeatInterval)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultSchedule.
     */
    public DefaultSchedule() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}