/*
 * Copyright 2019-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.yrtietftetypes.rev20160320.yrtietftetypes.recordroutesubobject.type;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.yrtietftetypes.rev20160320.yrtietftetypes.recordroutesubobject.Type;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of label.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultLabel extends InnerModelObject implements Label {
    protected long value;
    protected short flags;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public long value() {
        return value;
    }

    @Override
    public short flags() {
        return flags;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void value(long value) {
        valueLeafFlags.set(LeafIdentifier.VALUE.getLeafIndex());
        this.value = value;
    }

    @Override
    public void flags(short flags) {
        valueLeafFlags.set(LeafIdentifier.FLAGS.getLeafIndex());
        this.flags = flags;
    }

    @Override
    public int hashCode() {
        return Objects.hash(value, flags, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultLabel) {
            DefaultLabel other = (DefaultLabel) obj;
            return
                Objects.deepEquals(value, other.value) &&
                Objects.deepEquals(flags, other.flags) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("value=" + value)
            .add("flags=" + flags)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultLabel.
     */
    public DefaultLabel() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}