package org.openbp.cockpit.modeler.util;

import org.openbp.common.ExceptionUtil;
import org.openbp.core.CoreConstants;
import org.openbp.core.model.ModelException;
import org.openbp.core.model.ModelQualifier;
import org.openbp.core.model.item.ItemTypes;
import org.openbp.core.model.item.process.ActivityNodeImpl;
import org.openbp.core.model.item.process.DecisionNodeImpl;
import org.openbp.core.model.item.process.FinalNodeImpl;
import org.openbp.core.model.item.process.ForkNodeImpl;
import org.openbp.core.model.item.process.InitialNodeImpl;
import org.openbp.core.model.item.process.JoinNodeImpl;
import org.openbp.core.model.item.process.MergeNodeImpl;
import org.openbp.core.model.item.process.MultiSocketNode;
import org.openbp.core.model.item.process.Node;
import org.openbp.core.model.item.process.NodeParam;
import org.openbp.core.model.item.process.NodeParamImpl;
import org.openbp.core.model.item.process.NodeSocket;
import org.openbp.core.model.item.process.NodeSocketImpl;
import org.openbp.core.model.item.process.PlaceholderNodeImpl;
import org.openbp.core.model.item.process.ProcessUtil;
import org.openbp.core.model.item.process.VisualNodeImpl;
import org.openbp.core.model.item.process.WaitStateNodeImpl;
import org.openbp.core.model.item.process.WorkflowEndNodeImpl;
import org.openbp.core.model.item.process.WorkflowNodeImpl;
import org.openbp.core.model.item.type.DataTypeItem;
import org.openbp.guiclient.model.ModelConnector;

/**
 * Process element factory.
 * This class is a singleton.
 *
 * @author Heiko Erhardt
 */
public class ProcessElementFactory
{
	public static final String SYSTEM_TYPE_OBJECT = "Object";
	public static final String SYSTEM_TYPE_STRING = "String";
	public static final String SYSTEM_TYPE_INTEGER = "Integer";
	public static final String SYSTEM_TYPE_WORKFLOWTASK = "WorkflowTask";

	// TODO FIX 5 This should become a Spring-managed bean...
	/** Singleton instance */
	private static ProcessElementFactory singletonInstance;

	/**
	 * Gets the singleton instance of this class.
	 */
	public static synchronized ProcessElementFactory getInstance()
	{
		if (singletonInstance == null)
			singletonInstance = new ProcessElementFactory();
		return singletonInstance;
	}

	/**
	 * Private constructor.
	 */
	private ProcessElementFactory()
	{
	}

	/**
	 * Creates a standard initial node.
	 *
	 * @return The new node
	 */
	public InitialNodeImpl createStandardInitialNode()
	{
		InitialNodeImpl node = new InitialNodeImpl();
		node.setName(CoreConstants.DEFAULT_INITIAL_NODE_NAME);

		NodeSocket socket = createExitSocket(CoreConstants.SOCKET_OUT, true);
		socket.setGeometry(ProcessUtil.createSocketGeometry("s"));
		node.setSocket(socket);

		return node;
	}

	/**
	 * Creates a standard final node.
	 *
	 * @return The new node
	 */
	public FinalNodeImpl createStandardFinalNode()
	{
		FinalNodeImpl node = new FinalNodeImpl();
		node.setName(CoreConstants.DEFAULT_FINAL_NODE_NAME);

		NodeSocket socket = createEntrySocket(CoreConstants.SOCKET_IN, true);
		socket.setGeometry(ProcessUtil.createSocketGeometry("n"));
		node.setSocket(socket);

		return node;
	}

	/**
	 * Creates a standard decision node.
	 *
	 * @return The new node
	 */
	public DecisionNodeImpl createStandardDecisionNode()
	{
		DecisionNodeImpl node = new DecisionNodeImpl();
		createStandardName(node);

		NodeSocket inSocket = createEntrySocket(CoreConstants.DEFAULT_INITIAL_NODE_NAME, true);
		NodeSocket yesSocket = createExitSocket(CoreConstants.SOCKET_YES, true);
		NodeSocket noSocket = createExitSocket(CoreConstants.SOCKET_NO, false);

		inSocket.setGeometry(ProcessUtil.createSocketGeometry("n"));
		yesSocket.setGeometry(ProcessUtil.createSocketGeometry("s"));
		noSocket.setGeometry(ProcessUtil.createSocketGeometry("e"));

		node.addSocket(inSocket);
		node.addSocket(yesSocket);
		node.addSocket(noSocket);

		return node;
	}

	/**
	 * Creates a standard fork node.
	 *
	 * @return The new node
	 */
	public ForkNodeImpl createStandardForkNode()
	{
		ForkNodeImpl node = new ForkNodeImpl();
		createStandardName(node);

		NodeSocket inSocket = createEntrySocket(CoreConstants.DEFAULT_INITIAL_NODE_NAME, true);
		NodeSocket outSocket = createExitSocket(CoreConstants.DEFAULT_FINAL_NODE_NAME, true);

		createNodeParam(CoreConstants.FORK_COLLECTION_PARAM, "Collection", "If present, for each element in the collection, a separate token will be created. The element can be retrieved at the outgoing token of the fork node using the CollectionElement parameter.", SYSTEM_TYPE_OBJECT, false, true, inSocket);
		createNodeParam(CoreConstants.FORK_COLLECTION_ELEMENT_PARAM, "Collection element", "Collection element that spawned the token.\nNote that the leaving token can run in non-interactive mode only. No visuals may be displayed here.", SYSTEM_TYPE_OBJECT, false, true, outSocket);

		node.addSocket(inSocket);
		node.addSocket(outSocket);

		return node;
	}

	/**
	 * Creates a standard join node.
	 *
	 * @return The new node
	 */
	public JoinNodeImpl createStandardJoinNode()
	{
		JoinNodeImpl node = new JoinNodeImpl();
		configureStandardNode(node);
		return node;
	}

	/**
	 * Creates a standard wait state node.
	 *
	 * @return The new node
	 */
	public WaitStateNodeImpl createStandardWaitStateNode()
	{
		WaitStateNodeImpl node = new WaitStateNodeImpl();
		configureStandardNode(node);
		return node;
	}

	/**
	 * Creates a standard placeholder node.
	 *
	 * @return The new node
	 */
	public PlaceholderNodeImpl createStandardPlaceholderNode()
	{
		PlaceholderNodeImpl node = new PlaceholderNodeImpl();
		configureStandardNode(node);
		return node;
	}

	/**
	 * Creates a standard activity node.
	 *
	 * @return The new node
	 */
	public ActivityNodeImpl createStandardActivityNode()
	{
		ActivityNodeImpl node = new ActivityNodeImpl();
		configureStandardNode(node);
		return node;
	}

	/**
	 * Creates a standard visual node.
	 *
	 * @return The new node
	 */
	public VisualNodeImpl createStandardVisualNode()
	{
		VisualNodeImpl node = new VisualNodeImpl();
		configureStandardNode(node);
		return node;
	}

	/**
	 * Creates a standard workflow node.
	 *
	 * @return The new node
	 */
	public WorkflowNodeImpl createStandardWorkflowNode()
	{
		WorkflowNodeImpl node = new WorkflowNodeImpl();

		createStandardName(node);

		NodeSocket inSocket = createEntrySocket(CoreConstants.DEFAULT_INITIAL_NODE_NAME, true);
		NodeSocket publishedSocket = createExitSocket(CoreConstants.SOCKET_TASK_PUBLISHED, false);
		NodeSocket acceptedSocket = createExitSocket(CoreConstants.SOCKET_TASK_ACCEPTED, true);

		// Create some (by default hidden and optional) workflow parameters at the 'In' socket
		createNodeParam("StepName", "Step name", "System name of this workflow step for dynamic assignment", SYSTEM_TYPE_STRING, false, true, inSocket);
		createNodeParam("StepDisplayName", "Step title", "Title of this workflow step for dynamic assignment", SYSTEM_TYPE_STRING, false, true, inSocket);
		createNodeParam("StepDescription", "Step description", "Detailled description of this workflow step for dynamic assignment", SYSTEM_TYPE_STRING, false, true, inSocket);
		createNodeParam("RoleId", "Role id", "Id of the role this workflow task is assigned to (public worklist) for dynamic assignment", SYSTEM_TYPE_STRING, false, true, inSocket);
		createNodeParam("UserId", "User id", "Id of the user this workflow task is assigned to (private worklist) for dynamic assignment", SYSTEM_TYPE_STRING, false, true, inSocket);
		createNodeParam("Priority", "Priority", "Priority of this workflow task (1-5) for dynamic assignment", SYSTEM_TYPE_INTEGER, false, true, inSocket);
		createNodeParam("RoleDefinitions", "Role definitions", "XML string describing ad-hoc role definitions that may override standard user-role assignments", SYSTEM_TYPE_STRING, false, true, inSocket);
		createNodeParam("DocumentIds", "Document ids", "List of ids (strings) or a single id of the documents or the dodcument that is processed by this workflow", SYSTEM_TYPE_OBJECT, false, true, inSocket);
		createNodeParam("DocumentMode", "Document mode", "Identifier that describes the processing mode of the document (application-specific, i. e. 'read', 'write' etc.", SYSTEM_TYPE_STRING, false, true, inSocket);

		// Create the hidden and optional WorkflowTask parameter at all sockets
		createNodeParam("WorkflowTask", "Workflow task", "Workflow task in case of reassignment of an existing task", SYSTEM_TYPE_WORKFLOWTASK, false, true, inSocket);
		createNodeParam("WorkflowTask", "Workflow task", "Created workflow task", SYSTEM_TYPE_WORKFLOWTASK, false, true, publishedSocket);
		createNodeParam("WorkflowTask", "Workflow task", "Resumed workflow task", SYSTEM_TYPE_WORKFLOWTASK, false, true, acceptedSocket);

		node.addSocket(inSocket);
		node.addSocket(publishedSocket);
		node.addSocket(acceptedSocket);

		return node;
	}

	/**
	 * Creates a standard workflow end node.
	 *
	 * @return The new node
	 */
	public WorkflowEndNodeImpl createStandardWorkflowEndNode()
	{
		WorkflowEndNodeImpl node = new WorkflowEndNodeImpl();

		createStandardName(node);

		NodeSocket inSocket = createEntrySocket(CoreConstants.DEFAULT_INITIAL_NODE_NAME, true);

		// Create the hidden and optional WorkflowTask parameter
		createNodeParam("WorkflowTask", "Workflow task", "Workflow task that identifies the current workflow group or null to end all workflows that run in the current workflow group", SYSTEM_TYPE_WORKFLOWTASK, false, true, inSocket);

		node.addSocket(inSocket);

		return node;
	}

	/**
	 * Ensures that the given socket contains a workflow task parameter and adds one if not present.
	 *
	 * @param socket Socket to check
	 */
	public void ensureWorkflowTaskParameter(NodeSocket socket)
	{
		if (socket.getParamByName("WorkflowTask") == null)
		{
			createNodeParam("WorkflowTask", "Workflow task", "Resumed workflow task", SYSTEM_TYPE_WORKFLOWTASK, false, true, socket);
		}
	}

	/**
	 * Creates a standard merge node.
	 *
	 * @return The new node
	 */
	public MergeNodeImpl createStandardMergeNode()
	{
		MergeNodeImpl node = new MergeNodeImpl();
		createStandardName(node);

		NodeSocket inSocket1 = createEntrySocket(CoreConstants.DEFAULT_INITIAL_NODE_NAME + "1", true);
		NodeSocket inSocket2 = createEntrySocket(CoreConstants.DEFAULT_INITIAL_NODE_NAME + "2", false);
		NodeSocket outSocket = createExitSocket(CoreConstants.DEFAULT_FINAL_NODE_NAME, true);

		inSocket1.setGeometry(ProcessUtil.createSocketGeometry("n"));
		inSocket2.setGeometry(ProcessUtil.createSocketGeometry("e"));
		outSocket.setGeometry(ProcessUtil.createSocketGeometry("s"));

		node.addSocket(inSocket1);
		node.addSocket(inSocket2);
		node.addSocket(outSocket);

		return node;
	}

	/**
	 * Creates a new node entry socket.
	 *
	 * @param name Name of the socket
	 * @param isDefault true for a default socket
	 * @return The new socket
	 */
	public NodeSocket createEntrySocket(String name, boolean isDefault)
	{
		NodeSocket socket = new NodeSocketImpl();
		socket.setName(name);

		socket.setEntrySocket(true);
		socket.setDefaultSocket(isDefault);
		return socket;
	}

	/**
	 * Creates a new node entry socket.
	 *
	 * @param name Name of the socket
	 * @param isDefault true for a default socket
	 * @return The new socket
	 */
	public NodeSocket createExitSocket(String name, boolean isDefault)
	{
		NodeSocket socket = new NodeSocketImpl();
		socket.setName(name);

		socket.setEntrySocket(false);
		socket.setDefaultSocket(isDefault);
		return socket;
	}

	private void configureStandardNode(MultiSocketNode node)
	{
		createStandardName(node);

		NodeSocket inSocket = createEntrySocket(CoreConstants.DEFAULT_INITIAL_NODE_NAME, true);
		NodeSocket outSocket = createExitSocket(CoreConstants.DEFAULT_FINAL_NODE_NAME, true);

		node.addSocket(inSocket);
		node.addSocket(outSocket);
	}

	/**
	 * Creates a new node parameter.
	 *
	 * @param name Name
	 * @param displayName Display name
	 * @param description Description
	 * @param systemTypeName Type name of a data type of the system model
	 * @param visible
	 *		true	The node parameter is visible by default
	 *		false	The node parameter is hidden by default
	 * @param optional
	 *		true	This is an optional parameter.<br>
	 *		false	The parameter is required.
	 * @param socket Socket
	 * @return The new parameter
	 */
	private NodeParam createNodeParam(String name, String displayName, String description, String systemTypeName, boolean visible, boolean optional, NodeSocket socket)
	{
		NodeParam param = new NodeParamImpl();

		param.setName(name);
		param.setDisplayName(displayName);
		param.setDescription(description);

		ModelQualifier typeQualifier = new ModelQualifier(CoreConstants.SYSTEM_MODEL_NAME, systemTypeName, ItemTypes.TYPE, null);
		DataTypeItem dataType = null;
		try
		{
			dataType = (DataTypeItem) ModelConnector.getInstance().getItemByQualifier(typeQualifier, true);
		}
		catch (ModelException e)
		{
			ExceptionUtil.printTrace(e);
			return null;
		}
		param.setDataType(dataType);
		param.setTypeName(systemTypeName);

		param.setVisible(visible);
		param.setOptional(optional);

		if (socket != null)
		{
			socket.addParam(param);
		}

		return param;
	}

	private void createStandardName(Node node)
	{
		String s = node.getClass().getName();
		int i = s.lastIndexOf('.');
		s = s.substring (i + 1);
		i = s.lastIndexOf("Node");
		if (i > 0)
			s = s.substring(0, i);
		node.setName(s);
	}
}
