/*
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 */
package org.openbp.cockpit.modeler.figures.generic;

import java.awt.Color;
import java.text.DecimalFormat;
import java.util.StringTokenizer;

/**
 * Utility methods for geometry data encoding/decoding.
 *
 * @author Stephan Pauxberger
 */
public final class GeometryUtil
{
	/**
	 * Private constructor prevents instantiation.
	 */
	private GeometryUtil()
	{
	}

	/**
	 * Parses an integer value.
	 *
	 * @param st Tokenizer providing the geometry parameter stream
	 * @param paramName Name of the current parameter
	 * @param objectName Name of the process object that owns the parameter
	 * @return The integer
	 */
	public static int parseInt(StringTokenizer st, String paramName, String objectName)
	{
		try
		{
			if (! st.hasMoreTokens())
				throw new GeometryException("Missing value for paramter '" + paramName + "'.", objectName);
			String s = st.nextToken();
			return Integer.parseInt(s);
		}
		catch (NumberFormatException e)
		{
			throw new GeometryException("Invalid number format for paramter '" + paramName + "'.", objectName);
		}
	}

	/**
	 * Parses a double value.
	 *
	 * @param st Tokenizer providing the geometry parameter stream
	 * @param paramName Name of the current parameter
	 * @param objectName Name of the process object that owns the parameter
	 * @return The integer
	 */
	public static double parseDouble(StringTokenizer st, String paramName, String objectName)
	{
		try
		{
			if (! st.hasMoreTokens())
				throw new GeometryException("Missing value for paramter '" + paramName + "'.", objectName);
			String s = st.nextToken();
			return Double.parseDouble(s);
		}
		catch (NumberFormatException e)
		{
			throw new GeometryException("Invalid floating point number format for paramter '" + paramName + "'.", objectName);
		}
	}

	/**
	 * Parses a double value.
	 *
	 * @param s Geometry parameter
	 * @param paramName Name of the current parameter
	 * @param objectName Name of the process object that owns the parameter
	 * @return The integer
	 */
	public static double parseDouble(String s, String paramName, String objectName)
	{
		try
		{
			return Double.parseDouble(s);
		}
		catch (NumberFormatException e)
		{
			throw new GeometryException("Invalid floating point number format for paramter '" + paramName + "'.", objectName);
		}
	}

	/**
	 * Parses a RGB color value.
	 *
	 * @param st Tokenizer providing the geometry parameter stream
	 * @param paramName Name of the current parameter
	 * @param objectName Name of the process object that owns the parameter
	 * @return The colro
	 */
	public static Color parseColor(StringTokenizer st, String paramName, String objectName)
	{
		int r = parseInt(st, paramName, objectName);
		int g = parseInt(st, paramName, objectName);
		int b = parseInt(st, paramName, objectName);
		return new Color(r, g, b);
	}

	/**
	 * Parses an angle value.
	 *
	 * @param st Tokenizer providing the geometry parameter stream
	 * @param paramName Name of the current parameter
	 * @param objectName Name of the process object that owns the parameter
	 * @return The color
	 */
	public static double parseAngle(StringTokenizer st, String paramName, String objectName)
	{
		if (! st.hasMoreTokens())
			throw new GeometryException("Missing value for paramter '" + paramName + "'.", objectName);
		String s = st.nextToken();
		if (s.length() == 0)
			throw new GeometryException("Empty value for paramter '" + paramName + "'.", objectName);

		double angle;
		if (s.charAt(0) == 'd')
		{
			// Degree (radians); remove the indicating 'd' and convert ',' into '.'
			s = s.substring(1);
			s = s.replace(',', '.');
			double degree = parseDouble(s, paramName, objectName);
			angle = Math.toRadians(degree);
		}
		else
		{
			angle = parseDouble(s, paramName, objectName);
		}
		return angle;
	}

	private static DecimalFormat angleFormat = new DecimalFormat("#.##");

	/**
	 * Parses an angle value.
	 *
	 * @param angle Angle in radians
	 * @return The angle as degree specification
	 */
	public static String printAngle(double angle)
	{
		double degree = Math.toDegrees(angle);
		return "d" + angleFormat.format(degree);
	}
}
