/*
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 */
package org.openbp.cockpit.modeler.figures.process;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Stroke;
import java.awt.datatransfer.Transferable;
import java.awt.event.MouseEvent;
import java.util.List;
import java.util.StringTokenizer;
import java.util.Vector;

import org.openbp.cockpit.modeler.drawing.ProcessDrawing;
import org.openbp.cockpit.modeler.figures.VisualElement;
import org.openbp.cockpit.modeler.figures.VisualElementEvent;
import org.openbp.cockpit.modeler.figures.generic.GeometryException;
import org.openbp.cockpit.modeler.figures.generic.GeometryUtil;
import org.openbp.common.CollectionUtil;

import CH.ifa.draw.framework.Figure;
import CH.ifa.draw.framework.FigureChangeEvent;
import CH.ifa.draw.framework.FigureChangeListener;
import CH.ifa.draw.standard.AbstractFigure;

/**
 * Abstract line figure.
 *
 * @author Heiko Erhardt
 */
public abstract class LineFigure extends AbstractFigure
	implements VisualElement
{
	/** Vertical line */
	private boolean verticalLine;

	/** Stroke */
	private Stroke stroke;

	/** Color */
	private Color color;

	/** Position of the line (x coordinate) */
	protected int xPos;

	/** Position of the line (y coordinate) */
	protected int yPos;

	/** The visual status as defined in org.openbp.cockpit.modeler.figures.VisualElement */
	private int visualStatus = VISUAL_VISIBLE;

	/** Process drawing we belong to */
	private ProcessDrawing drawing;

	/**
	 * Default constructor.
	 *
	 * @param drawing Process drawing we belong to
	 */
	public LineFigure(ProcessDrawing drawing)
	{
		this.drawing = drawing;
	}

	/**
	 * Gets the vertical line.
	 */
	public boolean isVerticalLine()
	{
		return verticalLine;
	}

	/**
	 * Sets the vertical line.
	 */
	public void setVerticalLine(boolean verticalLine)
	{
		this.verticalLine = verticalLine;
	}

	/**
	 * Gets the stroke.
	 */
	public Stroke getStroke()
	{
		return stroke;
	}

	/**
	 * Sets the stroke.
	 */
	public void setStroke(Stroke stroke)
	{
		this.stroke = stroke;
	}

	/**
	 * Gets the color.
	 */
	public Color getColor()
	{
		return color;
	}

	/**
	 * Sets the color.
	 */
	public void setColor(Color color)
	{
		this.color = color;
	}

	//////////////////////////////////////////////////
	// @@ AbstractFigure overrides
	//////////////////////////////////////////////////

	/**
	 * We display a single font size handle in the top left corner only.
	 */
	public Vector handles()
	{
		return CollectionUtil.EMPTY_VECTOR;
	}

	public void basicDisplayBox(Point origin, Point corner)
	{
		xPos = origin.x;
		if (xPos < 0)
			xPos = 0;
		yPos = origin.y;
		if (yPos < 0)
			yPos = 0;
	}

	public Rectangle displayBox()
	{
		// We keep the display box small in order not to affect the entire drawing size
		// Instead, we override invalidate() in derived classes
		return new Rectangle(xPos - 1, yPos - 1, 3, 3);
	}

	public void invalidate()
	{
		FigureChangeListener l = listener();
		if (l != null)
		{
			Rectangle r = infiniteDisplayBox();
			l.figureInvalidated(new FigureChangeEvent(this, r));
		}
	}

	protected void basicMoveBy(int x, int y)
	{
		xPos += x;
		if (xPos < 0)
			xPos = 0;
		yPos += y;
		if (yPos < 0)
			yPos = 0;
	}

	/**
	 * Returns the display box of the figure that extends until infinity (or at least up to a very high value).
	 * @return The display box
	 */
	public Rectangle infiniteDisplayBox()
	{
		Rectangle r;
		if (verticalLine)
			r = new Rectangle(xPos - 1, 0, 3, 10000);
		else
			r = new Rectangle(0, yPos - 1, 10000, 3);
		return r;
	}

	public boolean containsPoint(int x, int y)
	{
		if (verticalLine)
			return x >= xPos - 4 && x <= xPos + 4;
		else
			return y >= yPos - 4 && y <= yPos + 4;
	}

	public void draw(Graphics g)
	{
		Graphics2D g2 = (Graphics2D) g;

		Color oldColor = g2.getColor();
		g2.setColor(getColor());

		Stroke oldStroke = g2.getStroke();
		g2.setStroke(getStroke());

		if (verticalLine)
			g.drawLine(xPos, 0, xPos, 10000);
		else
			g.drawLine(0, yPos, 10000, yPos);

		g2.setColor(oldColor);
		g2.setStroke(oldStroke);
	}

	//////////////////////////////////////////////////
	// @@ Geometry serialization support
	//////////////////////////////////////////////////

	/**
	 * Decodes enclosed geometry information.
	 *
	 * @param geometry Geometry string to decode or null
	 */
	public void decodeGeometry(String geometry)
	{
		if (geometry == null)
			return;

		String errIdent = "line figure";
		StringTokenizer st = new StringTokenizer(geometry, ":");
		while (st.hasMoreTokens())
		{
			String paramName = st.nextToken();
			if (paramName.equalsIgnoreCase("x"))
			{
				xPos = GeometryUtil.parseInt(st, paramName, errIdent);
			}
			else if (paramName.equalsIgnoreCase("y"))
			{
				yPos = GeometryUtil.parseInt(st, paramName, errIdent);
			}
			else
			{
				throw new GeometryException("Unknown paramter '" + paramName + "'.", errIdent);
			}
		}
	}

	/**
	 * Encodes the figure geometry in a string.
	 *
	 * @return Geometry string containing "|" and ":" separated tokens
	 */
	public String encodeGeometry()
	{
		return "x:" + xPos + ":y:" + yPos;
	}

	//////////////////////////////////////////////////
	// @@ VisualElement implementation
	//////////////////////////////////////////////////

	public void setDrawing(ProcessDrawing drawing)
	{
		this.drawing = drawing;
	}

	public ProcessDrawing getDrawing()
	{
		return drawing;
	}

	public VisualElement getParentElement()
	{
		return getDrawing();
	}

	public Figure getPresentationFigure()
	{
		return null;
	}

	public void updatePresentationFigure()
	{
		// No dynamic presentation figure, so do nothing
	}

	public boolean isVisible()
	{
		return (visualStatus & VisualElement.VISUAL_VISIBLE) != 0;
	}

	public void setVisible(boolean visible)
	{
		willChange();

		if (visible)
		{
			visualStatus |= VisualElement.VISUAL_VISIBLE;
		}
		else
		{
			visualStatus &= ~VisualElement.VISUAL_VISIBLE;
		}

		changed();
	}

	public boolean handleEvent(VisualElementEvent event)
	{
		return false;
	}

	/////////////////////////////////////////////////////////////////////////
	// @@ InteractionClient implementation
	// Dropping not allowed, nevertheless we have to implement it.
	/////////////////////////////////////////////////////////////////////////

	public void dragActionTriggered(Object regionId, Point p)
	{
	}

	public void dragEnded(Transferable transferable)
	{
	}

	public void dragStarted(Transferable transferable)
	{
	}

	public List getAllDropRegions(List flavors, Transferable data, MouseEvent mouseEvent)
	{
		return getDropRegions(flavors, data, mouseEvent);
	}

	public List getDropRegions(List flavors, Transferable data, MouseEvent mouseEvent)
	{
		return null;
	}

	public List getImportersAt(Point p)
	{
		return null;
	}

	public List getAllImportersAt(Point p)
	{
		return null;
	}

	public List getSubClients()
	{
		return null;
	}

	public boolean importData(Object regionId, Transferable data, Point p)
	{
		return false;
	}
}
