/*
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 */
package org.openbp.cockpit.modeler.skins;

import java.util.List;

import javax.swing.ImageIcon;

import org.openbp.cockpit.modeler.figures.generic.XFigureDescriptor;
import org.openbp.common.MsgFormat;

/**
 * Symbol descriptor.
 *
 * @author Heiko Erhardt
 */
public class SymbolDescriptor extends XFigureDescriptor
{
	/** Flag: Image display disabled */
	public static final int FLAG_IMAGE_DISABLED = (1 << 0);

	/** Symbol type */
	private String symbolType;

	/** Flags */
	private int flags;

	/** Text position */
	private String textPosition;

	/** Overlay position */
	private String overlayPosition;

	/** Overlay resource key */
	private String overlayResourceKey;

	/** Optional image */
	private transient ImageIcon overlayIcon;

	/**
	 * Default constructor.
	 */
	public SymbolDescriptor()
	{
	}

	//////////////////////////////////////////////////
	// @@ XFigureDescriptor overrides
	//////////////////////////////////////////////////

	/**
	 * Adds an error msg, prepending it by the node type name.
	 *
	 * @param errorMsgs List of strings that holds error messages
	 * @param errorMsg Error msg to add
	 */
	protected void addErrorMsg(List errorMsgs, String errorMsg)
	{
		errorMsgs.add(MsgFormat.format("Figure $0: $1", symbolType, errorMsg));
	}

	/**
	 * Gets the symbol type.
	 */
	public String getSymbolType()
	{
		return symbolType;
	}

	/**
	 * Sets the symbol type.
	 */
	public void setSymbolType(String symbolType)
	{
		this.symbolType = symbolType;
	}

	/**
	 * Gets the image disabled.
	 */
	public boolean isImageDisabled()
	{
		return (flags & FLAG_IMAGE_DISABLED) != 0;
	}

	/**
	 * Sets the image disabled.
	 */
	public void setImageDisabled(boolean imageDisabled)
	{
		if (imageDisabled)
		{
			flags |= FLAG_IMAGE_DISABLED;
		}
		else
		{
			flags &= ~FLAG_IMAGE_DISABLED;
		}
	}

	/**
	 * Gets the text position.
	 */
	public String getTextPosition()
	{
		return textPosition;
	}

	/**
	 * Sets the text position.
	 */
	public void setTextPosition(String textPosition)
	{
		this.textPosition = textPosition;
	}

	/**
	 * Gets the overlay position.
	 */
	public String getOverlayPosition()
	{
		return overlayPosition;
	}

	/**
	 * Sets the overlay position.
	 */
	public void setOverlayPosition(String overlayPosition)
	{
		this.overlayPosition = overlayPosition;
	}

	/**
	 * Gets the overlay resource key.
	 */
	public String getOverlayResourceKey()
	{
		return overlayResourceKey;
	}

	/**
	 * Sets the overlay resource key.
	 */
	public void setOverlayResourceKey(String overlayResourceKey)
	{
		this.overlayResourceKey = overlayResourceKey;
	}

	//////////////////////////////////////////////////
	// @@ Data access
	//////////////////////////////////////////////////

	/**
	 * Gets the optional image.
	 */
	public ImageIcon getOverlayIcon()
	{
		return overlayIcon;
	}

	/**
	 * Sets the optional image.
	 */
	public void setOverlayIcon(ImageIcon overlayIcon)
	{
		this.overlayIcon = overlayIcon;
	}
}
