/**
 * Licensed to the Austrian Association for Software Tool Integration (AASTI)
 * under one or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information regarding copyright
 * ownership. The AASTI licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.openengsb.domain.report;

import java.util.List;

import org.openengsb.core.api.Domain;
import org.openengsb.core.api.Event;
import org.openengsb.domain.report.model.Report;
import org.openengsb.domain.report.model.ReportPart;

public interface ReportDomain extends Domain {

    /**
     * Generate the report using the information stored under the given reportId and stop collecting data for this
     * report. The report is added to the specified report category and named as specified by the {@code reportName}
     * parameter. The category is created if it is not already present. If there is already a report with the same name
     * under this category it is overwritten.
     *
     * @return the generated report
     * @throws NoSuchReportException if no report with the given {@code reportId} is currently collecting data.
     */
    Report generateReport(String reportId, String category, String reportName) throws NoSuchReportException;

    /**
     *
     * Generate a report based on the currently available data stored for the given {@code reportId}. The report is
     * generated but not stored. Furthermore the data collection is not stopped for this report. This method can be used
     * if a preview of the report is needed, but the data collection is not finished yet.
     *
     * @return the generated report draft
     * @throws NoSuchReportException if no report with the given {@code reportId} is currently collecting data.
     */
    Report getDraft(String reportId, String draftName) throws NoSuchReportException;

    /**
     * Start a report data collection. If the data collection process is finished the report can be generated by calling
     * the {@link #generateReport(String, String, String)} method specifying the reportId returned by this method.
     *
     * @return the reportId that can later be used to generate the report by calling
     *         {@link #generateReport(String, String, String)}
     */
    String collectData();

    /**
     * Add the given report part to the report data currently collected for the given {@code reportId}. The report part
     * is appended at the end of the report.
     *
     * @throws NoSuchReportException if no report with the given {@code reportId} is currently collecting data.
     */
    void addReportPart(String reportId, ReportPart reportPart) throws NoSuchReportException;

    /**
     * Analyzes the given event and adds all information stored in the event to the report data collection associated
     * with the given {@code reportId}, which was initialized by calling {@link #collectData(IdType, String)}.
     *
     * @throws NoSuchReportException if no report with the given {@code reportId} is currently collecting data
     */
    void processEvent(String reportId, Event event) throws NoSuchReportException;

    /**
     * Get all finished reports of the given category. Reports, which are currently generated and collect data are not
     * included. If the given category does not exist an empty list is returned.
     */
    List<Report> getAllReports(String category);

    /**
     * Store the given report in the report store under the given category. The category is created if it is not already
     * present. If there is already a report with the same name under this category it is overwritten.
     */
    void storeReport(String category, Report report);

    /**
     * Remove the given report from the given category. The report is deleted and cannot be restored if it is not also
     * stored under another category. If no such report exist no operation is performed.
     */
    void removeReport(String category, Report report);

    /**
     * Get all report categories.
     */
    List<String> getAllCategories();

    /**
     * Remove the given category and all reports stored in this category. If no category with the specified name exists,
     * nothing is done.
     */
    void removeCategory(String category);

    /**
     * Creates the given category, which is empty at the start. Reports can later be added by either calling
     * {@link #storeReport(String, Report)} or {@link #generateReport(String, String, String)} specifying the respective
     * category. If a category exists with the given name no operation is performed.
     */
    void createCategory(String category);
}
