/*
 * Decompiled with CFR 0.152.
 */
package org.openforis.collect.io.metadata.samplingpointdata;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.openforis.collect.metamodel.samplingdesign.SamplingPointGenerationSettings;
import org.openforis.collect.metamodel.samplingdesign.SamplingPointLevelGenerationSettings;
import org.openforis.collect.model.CollectSurvey;
import org.openforis.collect.model.SamplingDesignItem;
import org.openforis.idm.geospatial.CoordinateOperations;
import org.openforis.idm.model.Coordinate;

public class SamplingPointDataGenerator {
    private static final double DEFAULT_POINT_SIZE = 10.0;
    private CollectSurvey survey;
    private List<List<SamplingDesignItem>> samplingPointsByLevel;
    private SamplingPointGenerationSettings configuration;
    private CoordinateOperations coordinateOperations;

    public SamplingPointDataGenerator(CoordinateOperations coordinateOperations, CollectSurvey survey, List<List<SamplingDesignItem>> samplingPointsByLevel, SamplingPointGenerationSettings configuration) {
        this.coordinateOperations = coordinateOperations;
        this.survey = survey;
        this.samplingPointsByLevel = samplingPointsByLevel;
        this.configuration = configuration;
    }

    public List<SamplingDesignItem> generate() {
        Coordinate latLonAoiCenter = this.calculateAoiCenter();
        return this.generateItems(0, Collections.emptyList(), latLonAoiCenter);
    }

    private List<SamplingDesignItem> generateItems(int levelIdx, List<String> previousLevelKeys, Coordinate latLonAoiCenter) {
        if (this.samplingPointsByLevel != null && this.samplingPointsByLevel.size() > levelIdx && CollectionUtils.isNotEmpty((Collection)this.samplingPointsByLevel.get(levelIdx))) {
            ArrayList<SamplingDesignItem> items = new ArrayList<SamplingDesignItem>();
            List<SamplingDesignItem> itemsInLevel = this.samplingPointsByLevel.get(levelIdx);
            items.addAll(itemsInLevel);
            for (SamplingDesignItem item : itemsInLevel) {
                item.setSurveyId(this.survey.getId());
                if (levelIdx >= this.configuration.getLevelsSettings().size() - 1) continue;
                List itemKeys = item.getLevelCodes();
                items.addAll(this.generateItems(levelIdx + 1, itemKeys, item.getCoordinate()));
            }
            return items;
        }
        ArrayList<SamplingDesignItem> items = new ArrayList<SamplingDesignItem>();
        List<Coordinate> locations = this.generateLocations(latLonAoiCenter, levelIdx);
        for (int locationIdx = 0; locationIdx < locations.size(); ++locationIdx) {
            Coordinate webMercatorCenter = locations.get(locationIdx);
            Coordinate latLonCenter = this.reprojectFromWebMercatorToLatLon(webMercatorCenter);
            SamplingDesignItem item = new SamplingDesignItem();
            item.setSrsId("EPSG:4326");
            item.setSurveyId(this.survey.getId());
            ArrayList<String> itemKeys = new ArrayList<String>(previousLevelKeys);
            itemKeys.add(String.valueOf(locationIdx + 1));
            item.setLevelCodes(itemKeys);
            item.setX(latLonCenter.getX());
            item.setY(latLonCenter.getY());
            items.add(item);
            if (levelIdx >= this.configuration.getLevelsSettings().size() - 1) continue;
            items.addAll(this.generateItems(levelIdx + 1, itemKeys, latLonCenter));
        }
        return items;
    }

    public Coordinate calculateAoiCenter() {
        List aoiBoundary = this.configuration.getAoiBoundary();
        double[] latitudes = new double[aoiBoundary.size()];
        double[] longitudes = new double[aoiBoundary.size()];
        for (int i = 0; i < aoiBoundary.size(); ++i) {
            Coordinate coord = (Coordinate)aoiBoundary.get(i);
            latitudes[i] = coord.getY();
            longitudes[i] = coord.getX();
        }
        double minBoundaryLatitude = NumberUtils.min((double[])latitudes);
        double maxBoundaryLatitude = NumberUtils.max((double[])latitudes);
        double minBoundaryLongitude = NumberUtils.min((double[])longitudes);
        double maxBoundaryLongitude = NumberUtils.max((double[])longitudes);
        return new Coordinate(Double.valueOf(minBoundaryLongitude + (maxBoundaryLongitude - minBoundaryLongitude) / 2.0), Double.valueOf(minBoundaryLatitude + (maxBoundaryLatitude - minBoundaryLatitude) / 2.0), "EPSG:4326");
    }

    private double calculateAoiWidth(int levelIdx) {
        switch (levelIdx) {
            case 0: {
                List aoiBoundary = this.configuration.getAoiBoundary();
                List<Coordinate> reprojectedAoiBoundary = this.reprojectToWebMercator(aoiBoundary);
                double[] longitudes = new double[reprojectedAoiBoundary.size()];
                for (int i = 0; i < reprojectedAoiBoundary.size(); ++i) {
                    Coordinate c = reprojectedAoiBoundary.get(i);
                    longitudes[i] = c.getX();
                }
                double minBoundaryLongitude = NumberUtils.min((double[])longitudes);
                double maxBoundaryLongitude = NumberUtils.max((double[])longitudes);
                return maxBoundaryLongitude - minBoundaryLongitude;
            }
        }
        SamplingPointLevelGenerationSettings previousLevelConfiguration = (SamplingPointLevelGenerationSettings)this.configuration.getLevelsSettings().get(levelIdx - 1);
        return previousLevelConfiguration.getPointWidth();
    }

    private List<Coordinate> generateLocations(Coordinate latLonCenter, int levelIdx) {
        SamplingPointLevelGenerationSettings pointsConfiguration = (SamplingPointLevelGenerationSettings)this.configuration.getLevelsSettings().get(levelIdx);
        Coordinate webMercatorAoiCenter = this.reprojectFromLatLonToWebMercator(latLonCenter);
        switch (pointsConfiguration.getShape()) {
            case CIRCLE: {
                double areaWidth = this.calculateAoiWidth(levelIdx);
                return this.generateLocationsInCircle(webMercatorAoiCenter, areaWidth / 2.0, pointsConfiguration);
            }
            case SQUARE: {
                SamplingPointLevelGenerationSettings nextLevelPointsConfiguration = levelIdx == this.configuration.getLevelsSettings().size() - 1 ? null : (SamplingPointLevelGenerationSettings)this.configuration.getLevelsSettings().get(levelIdx + 1);
                Double pointWidth = nextLevelPointsConfiguration == null ? 10.0 : nextLevelPointsConfiguration.getPointWidth();
                List<Coordinate> boundary = levelIdx == 0 ? this.configuration.getAoiBoundary() : this.generateSquareBoundary(latLonCenter, pointWidth);
                return this.generateLocationsInSquare(webMercatorAoiCenter, boundary, pointsConfiguration);
            }
        }
        throw new IllegalArgumentException("Shape type not supported: " + pointsConfiguration.getShape());
    }

    private List<Coordinate> generateSquareBoundary(Coordinate latLonCenter, double width) {
        Coordinate webMercatorCenter = this.reprojectFromLatLonToWebMercator(latLonCenter);
        ArrayList<Coordinate> latLonBoundary = new ArrayList<Coordinate>(4);
        double halfWidth = width / 2.0;
        latLonBoundary.add(new Coordinate(Double.valueOf(webMercatorCenter.getX() - halfWidth), Double.valueOf(webMercatorCenter.getY() + halfWidth), "EPSG:3857"));
        latLonBoundary.add(new Coordinate(Double.valueOf(webMercatorCenter.getX() + halfWidth), Double.valueOf(webMercatorCenter.getY() + halfWidth), "EPSG:3857"));
        latLonBoundary.add(new Coordinate(Double.valueOf(webMercatorCenter.getX() - halfWidth), Double.valueOf(webMercatorCenter.getY() - halfWidth), "EPSG:3857"));
        latLonBoundary.add(new Coordinate(Double.valueOf(webMercatorCenter.getX() + halfWidth), Double.valueOf(webMercatorCenter.getY() - halfWidth), "EPSG:3857"));
        return this.reprojectToWebMercator(latLonBoundary);
    }

    private List<Coordinate> generateLocationsInCircle(Coordinate center, double circleRadius, SamplingPointLevelGenerationSettings c) {
        ArrayList<Coordinate> result = new ArrayList<Coordinate>(c.getNumPoints());
        double radiusSquared = circleRadius * circleRadius;
        double locationRadius = c.getPointWidth() / 2.0;
        switch (c.getDistribution()) {
            case RANDOM: {
                for (int i = 0; i < c.getNumPoints(); ++i) {
                    double offsetAngle = Math.random() * Math.PI * 2.0;
                    double offsetMagnitude = Math.random() * circleRadius;
                    double xOffset = offsetMagnitude * Math.cos(offsetAngle);
                    double yOffset = offsetMagnitude * Math.sin(offsetAngle);
                    double x = center.getX() + xOffset;
                    double y = center.getY() + yOffset;
                    result.add(new Coordinate(Double.valueOf(x), Double.valueOf(y), center.getSrsId()));
                }
                break;
            }
            case GRIDDED: {
                double left = center.getX() - circleRadius;
                double top = center.getY() - circleRadius;
                double numberOfSteps = Math.floor(2.0 * circleRadius / c.getResolution());
                int xStep = 0;
                while ((double)xStep < numberOfSteps) {
                    double x = left + (double)xStep * c.getResolution() + locationRadius;
                    int yStep = 0;
                    while ((double)yStep < numberOfSteps) {
                        double y = top + (double)yStep * c.getResolution() + locationRadius;
                        if (this.squareDistance(x, y, center.getX(), center.getY()) < radiusSquared) {
                            result.add(new Coordinate(Double.valueOf(x), Double.valueOf(y), center.getSrsId()));
                        }
                        ++yStep;
                    }
                    ++xStep;
                }
                break;
            }
        }
        return result;
    }

    private List<Coordinate> generateLocationsInSquare(Coordinate webMercatorAoiCenter, List<Coordinate> aoiBoundary, SamplingPointLevelGenerationSettings c) {
        ArrayList<Coordinate> result = new ArrayList<Coordinate>();
        List<Coordinate> reprojectedAoiBoundary = this.reprojectToWebMercator(aoiBoundary);
        double[] latitudes = new double[reprojectedAoiBoundary.size()];
        double[] longitudes = new double[reprojectedAoiBoundary.size()];
        for (int i = 0; i < reprojectedAoiBoundary.size(); ++i) {
            Coordinate coord = reprojectedAoiBoundary.get(i);
            latitudes[i] = coord.getY();
            longitudes[i] = coord.getX();
        }
        double pointRadius = c.getPointWidth() / 2.0;
        double minBoundaryLatitude = NumberUtils.min((double[])latitudes) + pointRadius;
        double maxBoundaryLatitude = NumberUtils.max((double[])latitudes) - pointRadius;
        double minBoundaryLongitude = NumberUtils.min((double[])longitudes) + pointRadius;
        double maxBoundaryLongitude = NumberUtils.max((double[])longitudes) - pointRadius;
        switch (c.getDistribution()) {
            case RANDOM: {
                for (int i = 0; i < c.getNumPoints(); ++i) {
                    double x = minBoundaryLongitude + (maxBoundaryLongitude - minBoundaryLongitude) * Math.random();
                    double y = minBoundaryLatitude + (maxBoundaryLatitude - minBoundaryLatitude) * Math.random();
                    result.add(new Coordinate(Double.valueOf(x), Double.valueOf(y), webMercatorAoiCenter.getSrsId()));
                }
                break;
            }
            case GRIDDED: {
                for (double lat = maxBoundaryLatitude; lat >= minBoundaryLatitude; lat -= c.getResolution()) {
                    for (double lon = minBoundaryLongitude; lon <= maxBoundaryLongitude; lon += c.getResolution()) {
                        result.add(new Coordinate(Double.valueOf(lon), Double.valueOf(lat), webMercatorAoiCenter.getSrsId()));
                    }
                }
                break;
            }
        }
        return result;
    }

    private double squareDistance(double x1, double y1, double x2, double y2) {
        return Math.pow(x2 - x1, 2.0) + Math.pow(y2 - y1, 2.0);
    }

    private List<Coordinate> reprojectToWebMercator(List<Coordinate> aoiBoundary) {
        ArrayList<Coordinate> reprojectedAoiBoundary = new ArrayList<Coordinate>(aoiBoundary.size());
        for (Coordinate coord : aoiBoundary) {
            Coordinate reprojectedCoord = this.reprojectFromLatLonToWebMercator(coord);
            reprojectedAoiBoundary.add(reprojectedCoord);
        }
        return reprojectedAoiBoundary;
    }

    private Coordinate reprojectFromLatLonToWebMercator(Coordinate coordinate) {
        return this.coordinateOperations.convertTo(coordinate, "EPSG:3857");
    }

    private Coordinate reprojectFromWebMercatorToLatLon(Coordinate coordinate) {
        return this.coordinateOperations.convertToWgs84(coordinate);
    }
}

