package org.opengis.cite.cat20.dgiwg10;

import static org.opengis.cite.cat20.dgiwg10.ErrorMessageKeys.XPATH_RESULT;
import static org.testng.Assert.assertTrue;

import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.xml.namespace.QName;
import javax.xml.transform.Source;
import javax.xml.validation.Validator;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.opengis.cite.cat20.dgiwg10.util.NamespaceBindings;
import org.opengis.cite.cat20.dgiwg10.util.XMLUtils;
import org.opengis.cite.validation.ValidationErrorHandler;
import org.testng.Assert;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.sun.jersey.api.client.ClientResponse;

/**
 * Provides a set of custom assertion methods.
 */
public class ETSAssert {

    private static final Logger LOGR = Logger.getLogger( ETSAssert.class.getPackage().getName() );

    private ETSAssert() {
    }

    /**
     * Asserts that the qualified name of a DOM Node matches the expected value.
     * 
     * @param node
     *            The Node to check.
     * @param qName
     *            A QName object containing a namespace name (URI) and a local part.
     */
    public static void assertQualifiedName( Node node, QName qName ) {
        Assert.assertEquals( node.getLocalName(), qName.getLocalPart(), ErrorMessage.get( ErrorMessageKeys.LOCAL_NAME ) );
        Assert.assertEquals( node.getNamespaceURI(), qName.getNamespaceURI(),
                             ErrorMessage.get( ErrorMessageKeys.NAMESPACE_NAME ) );
    }

    /**
     * Asserts that an XPath 1.0 expression holds true for the given evaluation context. The following standard
     * namespace bindings do not need to be explicitly declared:
     *
     * <ul>
     * <li>{@value org.opengis.cite.cat20.dgiwg10.Namespaces#OWS_PREFIX}:
     * {@value org.opengis.cite.cat20.dgiwg10.Namespaces#OWS}</li>
     * <li>{@value org.opengis.cite.cat20.dgiwg10.Namespaces#XLINK_PREFIX}:
     * {@value org.opengis.cite.cat20.dgiwg10.Namespaces#XLINK}</li>
     * <li>{@value org.opengis.cite.cat20.dgiwg10.Namespaces#CSW_PREFIX}:
     * {@value org.opengis.cite.cat20.dgiwg10.Namespaces#CSW}</li>
     * </ul>
     * 
     * @param expr
     *            A valid XPath 1.0 expression.
     * @param context
     *            The context node.
     * @param namespaceBindings
     *            A collection of namespace bindings for the XPath expression, where each entry maps a namespace URI
     *            (key) to a prefix (value). It may be {@code null}.
     */
    public static void assertXPath( String expr, Node context, Map<String, String> namespaceBindings ) {
        assertXPath( expr, context, namespaceBindings, null );
    }

    /**
     * Asserts that an XPath 1.0 expression holds true for the given evaluation context. The following standard
     * namespace bindings do not need to be explicitly declared:
     *
     * <ul>
     * <li>{@value org.opengis.cite.cat20.dgiwg10.Namespaces#OWS_PREFIX}:
     * {@value org.opengis.cite.cat20.dgiwg10.Namespaces#OWS}</li>
     * <li>{@value org.opengis.cite.cat20.dgiwg10.Namespaces#XLINK_PREFIX}:
     * {@value org.opengis.cite.cat20.dgiwg10.Namespaces#XLINK}</li>
     * <li>{@value org.opengis.cite.cat20.dgiwg10.Namespaces#CSW_PREFIX}:
     * {@value org.opengis.cite.cat20.dgiwg10.Namespaces#CSW}</li>
     * </ul>
     *
     * @param expr
     *            A valid XPath 1.0 expression.
     * @param context
     *            The context node.
     * @param namespaceBindings
     *            A collection of namespace bindings for the XPath expression, where each entry maps a namespace URI
     *            (key) to a prefix (value). It may be {@code null}.
     * @param assertionErrorMessage
     *            an optional message thrown if the assertion fails, if <code>null</code> the default message is used
     */
    public static void assertXPath( String expr, Node context, Map<String, String> namespaceBindings,
                                    String assertionErrorMessage ) {
        if ( null == context ) {
            throw new NullPointerException( "Context node is null." );
        }
        NamespaceBindings bindings = NamespaceBindings.withStandardBindings();
        bindings.addAllBindings( namespaceBindings );
        XPath xpath = XPathFactory.newInstance().newXPath();
        xpath.setNamespaceContext( bindings );
        Boolean result;
        try {
            result = (Boolean) xpath.evaluate( expr, context, XPathConstants.BOOLEAN );
        } catch ( XPathExpressionException xpe ) {
            String msg = ErrorMessage.format( ErrorMessageKeys.XPATH_ERROR, expr );
            LOGR.log( Level.WARNING, msg, xpe );
            throw new AssertionError( msg );
        }
        Element elemNode;
        if ( Document.class.isInstance( context ) ) {
            elemNode = Document.class.cast( context ).getDocumentElement();
        } else {
            elemNode = (Element) context;
        }
        String errorMessage = assertionErrorMessage;
        if ( errorMessage == null )
            errorMessage = ErrorMessage.format( XPATH_RESULT, elemNode.getNodeName(), expr );
        assertTrue( result, errorMessage );
    }

    /**
     * Asserts that an XML resource is schema-valid.
     * 
     * @param validator
     *            The Validator to use.
     * @param source
     *            The XML Source to be validated.
     */
    public static void assertSchemaValid( Validator validator, Source source ) {
        ValidationErrorHandler errHandler = new ValidationErrorHandler();
        validator.setErrorHandler( errHandler );
        try {
            validator.validate( source );
        } catch ( Exception e ) {
            throw new AssertionError( ErrorMessage.format( ErrorMessageKeys.XML_ERROR, e.getMessage() ) );
        }
        Assert.assertFalse( errHandler.errorsDetected(), ErrorMessage.format( ErrorMessageKeys.NOT_SCHEMA_VALID,
                                                                              errHandler.getErrorCount(),
                                                                              errHandler.toString() ) );
    }

    /**
     * Asserts that the given XML entity contains the expected number of descendant elements having the specified name.
     * 
     * @param xmlEntity
     *            A Document representing an XML entity.
     * @param elementName
     *            The qualified name of the element.
     * @param expectedCount
     *            The expected number of occurrences.
     */
    public static void assertDescendantElementCount( Document xmlEntity, QName elementName, int expectedCount ) {
        NodeList features = xmlEntity.getElementsByTagNameNS( elementName.getNamespaceURI(), elementName.getLocalPart() );
        Assert.assertEquals( features.getLength(), expectedCount,
                             String.format( "Unexpected number of %s descendant elements.", elementName ) );
    }

    /**
     * Asserts that the given response message contains an OGC exception report. The message body must contain an XML
     * document that has a document element with the following properties:
     *
     * <ul>
     * <li>[local name] = "ExceptionReport"</li>
     * <li>[namespace name] = "http://www.opengis.net/ows/2.0"</li>
     * </ul>
     *
     * @param rsp
     *            A ClientResponse object representing an HTTP response message.
     * @param exceptionCode
     *            The expected OGC exception code.
     * @param locator
     *            A case-insensitive string value expected to occur in the locator attribute (e.g. a parameter name);
     *            the attribute value will be ignored if the argument is null or empty.
     */
    public static void assertExceptionReport( ClientResponse rsp, String exceptionCode, String locator ) {
        Assert.assertEquals( rsp.getStatus(), ClientResponse.Status.BAD_REQUEST.getStatusCode(),
                             ErrorMessage.get( ErrorMessageKeys.UNEXPECTED_STATUS ) );
        Document doc = rsp.getEntity( Document.class );
        String expr = String.format( "//ows:Exception[@exceptionCode = '%s']", exceptionCode );
        NodeList nodeList = null;
        try {
            nodeList = XMLUtils.evaluateXPath( doc, expr, null );
        } catch ( XPathExpressionException xpe ) {
            // won't happen
        }
        assertTrue( nodeList.getLength() > 0, "Exception not found in response: " + expr );
        if ( null != locator && !locator.isEmpty() ) {
            Element exception = (Element) nodeList.item( 0 );
            String locatorValue = exception.getAttribute( "locator" ).toLowerCase();
            assertTrue( locatorValue.contains( locator.toLowerCase() ),
                        String.format( "Expected locator attribute to contain '%s']", locator ) );
        }
    }
}
