/*
 * Decompiled with CFR 0.152.
 */
package org.geotoolkit.nio;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.HashSet;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.event.EventListenerList;
import org.apache.sis.util.ArgumentChecks;
import org.geotoolkit.nio.PathChangeListener;
import org.geotoolkit.nio.PathChangedEvent;

public class DirectoryWatcher
implements Closeable {
    private static final Logger LOGGER = Logger.getLogger("org.geotoolkit.io");
    private final WatchService service;
    private final Thread watchThread;
    public final boolean recursive;
    protected final HashSet<Path> roots = new HashSet();
    protected final HashSet<Path> unregistered = new HashSet();
    protected DirectoryStream.Filter<Path> fileFilter = null;
    private final Object fileFilterLock = new Object();
    protected final EventListenerList listeners = new EventListenerList();
    private RegistrationErrorHandler registrationErrorHandler = DirectoryWatcher.logThenContinue();

    public DirectoryWatcher() throws IOException {
        this(false);
    }

    public DirectoryWatcher(boolean recursive) throws IOException {
        this.service = FileSystems.getDefault().newWatchService();
        this.recursive = recursive;
        this.watchThread = new Thread(new Runnable(){

            @Override
            public void run() {
                DirectoryWatcher.this.watch();
            }
        });
        this.watchThread.setDaemon(true);
    }

    public void setRegistrationErrorHandler(RegistrationErrorHandler handler) {
        this.registrationErrorHandler = handler == null ? DirectoryWatcher.logThenContinue() : handler;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setFileFilter(DirectoryStream.Filter<Path> filter) {
        Object object = this.fileFilterLock;
        synchronized (object) {
            this.fileFilter = filter;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DirectoryStream.Filter<Path> getFileFilter() {
        Object object = this.fileFilterLock;
        synchronized (object) {
            return this.fileFilter;
        }
    }

    public synchronized void register(Path ... paths) throws IOException, SecurityException {
        for (Path toWatch : paths) {
            if (!this.roots.contains(toWatch)) {
                try {
                    if (!Files.isDirectory(toWatch, new LinkOption[0])) {
                        Files.createDirectories(toWatch, new FileAttribute[0]);
                    }
                }
                catch (Exception e) {
                    this.registrationErrorHandler.handle(new RegistrationError(toWatch, true, e));
                }
                this.registerRoot(toWatch);
                this.roots.add(toWatch);
            }
            this.unregistered.remove(toWatch);
        }
    }

    public synchronized void unregister(Path ... paths) {
        for (Path dir : paths) {
            this.unregistered.add(dir);
            this.roots.remove(dir);
        }
    }

    private final void registerDir(Path dir) throws IOException, SecurityException {
        dir.register(this.service, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY);
        if (this.recursive) {
            Files.walkFileTree(dir, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult preVisitDirectory(Path subdir, BasicFileAttributes attrs) throws IOException {
                    subdir.register(DirectoryWatcher.this.service, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                    LOGGER.log(Level.FINE, "Cannot register a newly created directory", exc);
                    return FileVisitResult.SKIP_SUBTREE;
                }
            });
        }
    }

    private final void registerRoot(Path root) throws IOException, SecurityException {
        try {
            root.register(this.service, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY);
        }
        catch (Exception e) {
            this.registrationErrorHandler.handle(new RegistrationError(root, true, e));
        }
        if (this.recursive) {
            Files.walkFileTree(root, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult preVisitDirectory(Path subdir, BasicFileAttributes attrs) throws IOException {
                    try {
                        subdir.register(DirectoryWatcher.this.service, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY);
                    }
                    catch (Exception e) {
                        DirectoryWatcher.this.registrationErrorHandler.handle(new RegistrationError(subdir, false, e));
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                    DirectoryWatcher.this.registrationErrorHandler.handle(new RegistrationError(file, false, exc));
                    return FileVisitResult.SKIP_SUBTREE;
                }
            });
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void watch() {
        Thread currentThread = Thread.currentThread();
        while (!currentThread.isInterrupted()) {
            HashSet<Path> hashSet;
            WatchKey key;
            try {
                key = this.service.take();
            }
            catch (ClosedWatchServiceException e) {
                LOGGER.log(Level.INFO, "Folder watcher has been closed !");
                break;
            }
            catch (InterruptedException e) {
                LOGGER.log(Level.INFO, "Folder watcher has been interrupted !");
                currentThread.interrupt();
                break;
            }
            if (currentThread.isInterrupted()) break;
            if (!key.isValid()) continue;
            Path watchedPath = (Path)key.watchable();
            try {
                Path dereferenced = this.getUnregisteredParent(watchedPath);
                if (dereferenced != null) {
                    key.cancel();
                    if (!dereferenced.equals(watchedPath)) continue;
                    hashSet = this.unregistered;
                    synchronized (hashSet) {
                        this.unregistered.remove(dereferenced);
                        continue;
                    }
                }
                for (WatchEvent<?> event : key.pollEvents()) {
                    WatchEvent.Kind<?> kind = event.kind();
                    if (kind == StandardWatchEventKinds.OVERFLOW) {
                        LOGGER.log(Level.INFO, "Too many changes happened to the watched directory. Unable to catch them all.");
                        continue;
                    }
                    try {
                        boolean matchFileFilter;
                        Path target;
                        Object context = event.context();
                        if (context instanceof Path) {
                            target = (Path)context;
                        } else if (context instanceof File) {
                            target = ((File)context).toPath();
                        } else {
                            IllegalArgumentException e = new IllegalArgumentException("Watch event is of unknown type (need Path or File).");
                            LOGGER.log(Level.INFO, "Watch event skipped.", e);
                            continue;
                        }
                        target = watchedPath.resolve(target);
                        boolean isDirectory = Files.isDirectory(target, new LinkOption[0]);
                        if (isDirectory && this.recursive && kind == StandardWatchEventKinds.ENTRY_CREATE) {
                            try {
                                this.registerDir(target);
                            }
                            catch (IOException e) {
                                LOGGER.log(Level.WARNING, "Newly created folder cannot be watched : " + target, e);
                            }
                        }
                        Object object = this.fileFilterLock;
                        synchronized (object) {
                            matchFileFilter = this.fileFilter == null || this.fileFilter.accept(target);
                        }
                        if (!matchFileFilter) continue;
                        this.firePathChanged(target, kind, isDirectory, event.count());
                    }
                    catch (Exception e) {
                        LOGGER.log(Level.WARNING, "An error occurred while processing " + event.context() + " for the event " + kind, e);
                    }
                }
            }
            finally {
                if (key.reset()) continue;
                hashSet = this.roots;
                synchronized (hashSet) {
                    this.roots.remove(watchedPath);
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected Path getUnregisteredParent(Path target) {
        HashSet<Path> hashSet = this.unregistered;
        synchronized (hashSet) {
            for (Path path : this.unregistered) {
                if (!target.startsWith(path) && !path.equals(target)) continue;
                return path;
            }
        }
        return null;
    }

    public void start() {
        this.watchThread.start();
    }

    public void stop() {
        this.watchThread.interrupt();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public synchronized void close() throws IOException {
        this.stop();
        HashSet<Path> hashSet = this.roots;
        synchronized (hashSet) {
            this.roots.clear();
        }
        this.service.close();
    }

    public void addPathChangeListener(PathChangeListener listener) {
        ArgumentChecks.ensureNonNull("Event listener", listener);
        this.listeners.add(PathChangeListener.class, listener);
    }

    public void removePathChangeListener(PathChangeListener toForget) {
        ArgumentChecks.ensureNonNull("Event listener", toForget);
        this.listeners.remove(PathChangeListener.class, toForget);
    }

    protected void firePathChanged(Path target, WatchEvent.Kind kind, boolean isDirectory, int count) throws Exception {
        PathChangedEvent evt = new PathChangedEvent(this, target, kind, isDirectory, count);
        Exception traced = null;
        for (PathChangeListener l : (PathChangeListener[])this.listeners.getListeners(PathChangeListener.class)) {
            try {
                l.pathChanged(evt);
            }
            catch (Exception e) {
                if (traced == null) {
                    traced = e;
                    continue;
                }
                traced.addSuppressed(e);
            }
        }
        if (traced != null) {
            throw traced;
        }
    }

    public static RegistrationErrorHandler logThenContinue() {
        return event -> LOGGER.log(event.isRoot ? Level.WARNING : Level.FINE, event.error, () -> String.format("Cannot register %sdirectory: %s", event.isRoot ? "root " : "sub-", event.target));
    }

    public static RegistrationErrorHandler rethrow() {
        return DirectoryWatcher::rethrowError;
    }

    private static void rethrowError(RegistrationError info) throws IOException, RuntimeException {
        Exception error = info.error;
        if (error instanceof IOException) {
            throw (IOException)error;
        }
        if (error instanceof RuntimeException) {
            throw (RuntimeException)error;
        }
        throw new RuntimeException("Cannot register directory", error);
    }

    @FunctionalInterface
    public static interface RegistrationErrorHandler {
        public void handle(RegistrationError var1) throws IOException;
    }

    public final class RegistrationError {
        final Path target;
        final boolean isRoot;
        final Exception error;

        private RegistrationError(Path target, boolean isRoot, Exception error) {
            this.target = target;
            this.isRoot = isRoot;
            this.error = error;
        }
    }
}

