#!/bin/sh
#
# /etc/init.d/tomcat7 -- startup script for the Tomcat 7 servlet engine
#
# Written by Miquel van Smoorenburg <miquels@cistron.nl>.
# Modified for Debian GNU/Linux by Ian Murdock <imurdock@gnu.ai.mit.edu>.
# Modified for Tomcat by Stefan Gybas <sgybas@debian.org>.
# Modified for Tomcat6 by Thierry Carrez <thierry.carrez@ubuntu.com>.
# Modified for Tomcat7 by Ernesto Hernandez-Novich <emhn@itverx.com.ve>.
# Additional improvements by Jason Brittain <jason.brittain@mulesoft.com>.
#
### BEGIN INIT INFO
# Provides:          tomcat7
# Required-Start:    $local_fs $remote_fs $network
# Required-Stop:     $local_fs $remote_fs $network
# Should-Start:      $named
# Should-Stop:       $named
# Default-Start:     2 3 4 5
# Default-Stop:      0 1 6
# Short-Description: Start Tomcat.
# Description:       Start the Tomcat servlet engine.
### END INIT INFO

set -e

PATH=/bin:/usr/bin:/sbin:/usr/sbin
NAME=tomcat7
DESC="Tomcat servlet engine"
DEFAULT=/etc/default/$NAME
JVM_TMP=/tmp/tomcat7-$NAME-tmp

if [ `id -u` -ne 0 ]; then
    echo "You need root privileges to run this script"
    exit 1
fi
 
# Make sure tomcat is started with system locale
if [ -r /etc/default/locale ]; then
    . /etc/default/locale
    export LANG
fi

. /lib/lsb/init-functions

if [ -r /etc/default/rcS ]; then
    . /etc/default/rcS
fi


# The following variables can be overwritten in $DEFAULT

# Run Tomcat 7 as this user ID and group ID
TOMCAT7_USER=tomcat
TOMCAT7_GROUP=tomcat

# this is a work-around until there is a suitable runtime replacement 
# for dpkg-architecture for arch:all packages
# this function sets the variable JDK_DIRS
find_jdks()
{
    for java_version in 9 8 7 6
    do
        for jvmdir in /usr/lib/jvm/java-${java_version}-openjdk-* \
                      /usr/lib/jvm/jdk-${java_version}-oracle-* \
                      /usr/lib/jvm/jre-${java_version}-oracle-*
        do
            if [ -d "${jvmdir}" -a "${jvmdir}" != "/usr/lib/jvm/java-${java_version}-openjdk-common" ]
            then
                JDK_DIRS="${JDK_DIRS} ${jvmdir}"
            fi
        done
    done

    # Add older non multi arch installations
    JDK_DIRS="${JDK_DIRS} /usr/lib/jvm/java-6-openjdk /usr/lib/jvm/java-6-sun /usr/lib/jvm/java-7-oracle"
}

# The first existing directory is used for JAVA_HOME (if JAVA_HOME is not
# defined in $DEFAULT)
JDK_DIRS="/usr/lib/jvm/default-java"
find_jdks

# Look for the right JVM to use
for jdir in $JDK_DIRS; do
    if [ -r "$jdir/bin/java" -a -z "${JAVA_HOME}" ]; then
    JAVA_HOME="$jdir"
    fi
done
export JAVA_HOME

# Directory where the Tomcat 7 binary distribution resides
CATALINA_HOME=/opt/tomcat7

# Directory for per-instance configuration files and webapps
CATALINA_BASE=/opt/tomcat7

# Use the Java security manager? (yes/no)
TOMCAT7_SECURITY=no

# Default Java options
# Set java.awt.headless=true if JAVA_OPTS is not set so the
# Xalan XSL transformer can work without X11 display on JDK 1.4+
# It also looks like the default heap size of 64M is not enough for most cases
# so the maximum heap size is set to 1024M
if [ -z "$JAVA_OPTS" ]; then
    JAVA_OPTS="-Djava.awt.headless=true -server -Xmx1024M -XX:MaxPermSize=128m -DTE_BASE=/opt/te_base"
fi

# End of variables that can be overwritten in $DEFAULT

# overwrite settings from default file
if [ -f "$DEFAULT" ]; then
    . "$DEFAULT"
fi

if [ ! -f "$CATALINA_HOME/bin/bootstrap.jar" ]; then
    log_failure_msg "$NAME is not installed"
    exit 1
fi

if [ -z "$CATALINA_TMPDIR" ]; then
    CATALINA_TMPDIR="$JVM_TMP"
fi

# Set the JSP compiler if set in the tomcat7.default file
if [ -n "$JSP_COMPILER" ]; then
    JAVA_OPTS="$JAVA_OPTS -Dbuild.compiler=\"$JSP_COMPILER\""
fi

SECURITY=""
if [ "$TOMCAT7_SECURITY" = "yes" ]; then
    SECURITY="-security"
fi

# Define other required variables
CATALINA_PID="/var/run/$NAME.pid"
CATALINA_SH="$CATALINA_HOME/bin/catalina.sh"

# Look for Java Secure Sockets Extension (JSSE) JARs
if [ -z "${JSSE_HOME}" -a -r "${JAVA_HOME}/jre/lib/jsse.jar" ]; then
    JSSE_HOME="${JAVA_HOME}/jre/"
fi

catalina_sh() {
    # Escape any double quotes in the value of JAVA_OPTS
    JAVA_OPTS="$(echo $JAVA_OPTS | sed 's/\"/\\\"/g')"

    AUTHBIND_COMMAND=""
    if [ "$AUTHBIND" = "yes" -a "$1" = "start" ]; then
        AUTHBIND_COMMAND="/usr/bin/authbind --deep /bin/bash -c "
    fi

    # Define the command to run Tomcat's catalina.sh as a daemon
    # set -a tells sh to export assigned variables to spawned shells.
    TOMCAT_SH="set -a; JAVA_HOME=\"$JAVA_HOME\"; source \"$DEFAULT\"; \
        CATALINA_HOME=\"$CATALINA_HOME\"; \
        CATALINA_BASE=\"$CATALINA_BASE\"; \
        JAVA_OPTS=\"$JAVA_OPTS\"; \
        CATALINA_PID=\"$CATALINA_PID\"; \
        CATALINA_TMPDIR=\"$CATALINA_TMPDIR\"; \
        LANG=\"$LANG\"; JSSE_HOME=\"$JSSE_HOME\"; \
        cd \"$CATALINA_BASE\"; \
        \"$CATALINA_SH\" $@"

    if [ "$AUTHBIND" = "yes" -a "$1" = "start" ]; then
        TOMCAT_SH="'$TOMCAT_SH'"
    fi

    # Run the catalina.sh script as a daemon
    set +e
    if [ ! -f "$CATALINA_BASE"/logs/catalina.out ]; then
        # run install as tomcat7 to work around #841371
        su $TOMCAT7_USER -s /bin/bash -c "install -m 644 /dev/null $CATALINA_BASE/logs/catalina.out"
    fi
    install -o $TOMCAT7_USER -g adm -m 644 /dev/null "$CATALINA_PID"
    start-stop-daemon --start -b -u "$TOMCAT7_USER" -g "$TOMCAT7_GROUP" \
        -c "$TOMCAT7_USER" -d "$CATALINA_TMPDIR" -p "$CATALINA_PID" \
        -x /bin/bash -- -c "$AUTHBIND_COMMAND $TOMCAT_SH"
    status="$?"
    set +a -e
    return $status
}

case "$1" in
  start)
    if [ -z "$JAVA_HOME" ]; then
        log_failure_msg "no JDK or JRE found - please set JAVA_HOME"
        exit 1
    fi

    if [ ! -d "$CATALINA_BASE/conf" ]; then
        log_failure_msg "invalid CATALINA_BASE: $CATALINA_BASE"
        exit 1
    fi

    log_daemon_msg "Starting $DESC" "$NAME"
    if start-stop-daemon --test --start --pidfile "$CATALINA_PID" \
        --user $TOMCAT7_USER --exec "$JAVA_HOME/bin/java" \
        >/dev/null; then

        # Remove / recreate JVM_TMP directory
        rm -rf "$JVM_TMP"
        mkdir "$JVM_TMP" || {
            log_failure_msg "could not create JVM temporary directory"
            exit 1
        }
        chown -h $TOMCAT7_USER "$JVM_TMP"

        catalina_sh start $SECURITY
        sleep 5
            if start-stop-daemon --test --start --pidfile "$CATALINA_PID" \
            --user $TOMCAT7_USER --startas "$JAVA_HOME/bin/java" \
            >/dev/null; then
            if [ -f "$CATALINA_PID" ]; then
                rm -f "$CATALINA_PID"
            fi
            log_end_msg 1
        else
            log_end_msg 0
        fi
    else
            log_progress_msg "(already running)"
        log_end_msg 0
    fi
    ;;
  stop)
    log_daemon_msg "Stopping $DESC" "$NAME"

    set +e
    if [ -f "$CATALINA_PID" ]; then 
        start-stop-daemon --stop --pidfile "$CATALINA_PID" \
            --user "$TOMCAT7_USER" \
            --retry=TERM/20/KILL/5 >/dev/null
        if [ $? -eq 1 ]; then
            log_progress_msg "$DESC is not running but pid file exists, cleaning up"
        elif [ $? -eq 3 ]; then
            PID="`cat $CATALINA_PID`"
            log_failure_msg "Failed to stop $NAME (pid $PID)"
            exit 1
        fi
        rm -f "$CATALINA_PID"
        rm -rf "$JVM_TMP"
    else
        log_progress_msg "(not running)"
    fi
    log_end_msg 0
    set -e
    ;;
   status)
    set +e
    start-stop-daemon --test --start --pidfile "$CATALINA_PID" \
        --user $TOMCAT7_USER --exec "$JAVA_HOME/bin/java" \
        >/dev/null 2>&1
    if [ "$?" = "0" ]; then

        if [ -f "$CATALINA_PID" ]; then
            log_success_msg "$DESC is not running, but pid file exists."
            exit 1
        else
            log_success_msg "$DESC is not running."
            exit 3
        fi
    else
        log_success_msg "$DESC is running with pid `cat $CATALINA_PID`"
    fi
    set -e
        ;;
  restart|force-reload)
    if [ -f "$CATALINA_PID" ]; then
        $0 stop
        sleep 1
    fi
    $0 start
    ;;
  try-restart)
        if start-stop-daemon --test --start --pidfile "$CATALINA_PID" \
        --user $TOMCAT7_USER --exec "$JAVA_HOME/bin/java" \
        >/dev/null; then
        $0 start
    fi
        ;;
  *)
    log_success_msg "Usage: $0 {start|stop|restart|try-restart|force-reload|status}"
    exit 1
    ;;
esac

exit 0
