package org.opengis.cite.wfs11.util;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.xerces.xs.XSComplexTypeDefinition;
import org.apache.xerces.xs.XSConstants;
import org.apache.xerces.xs.XSElementDeclaration;
import org.apache.xerces.xs.XSModel;
import org.apache.xerces.xs.XSTypeDefinition;
import org.opengis.cite.iso19136.util.XMLSchemaModelUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

/**
 * <p>
 * XmlUtils class.
 * </p>
 *
 */
public class XmlUtils {

	/**
	 * Creates a {@link org.w3c.dom.Node} out of a {@link java.lang.String}
	 * @param xmlString never <code>null</code>
	 * @return the string as node, never <code>null</code>
	 * @throws java.lang.Exception if any.
	 */
	public static Node asNode(String xmlString) throws Exception {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		factory.setNamespaceAware(true);
		DocumentBuilder builder = factory.newDocumentBuilder();
		ByteArrayInputStream is = new ByteArrayInputStream(xmlString.getBytes());
		try {
			Document wfsCapabilities = builder.parse(is);
			return wfsCapabilities.getDocumentElement();
		}
		finally {
			is.close();
		}
	}

	/**
	 * Creates a {@link java.lang.String} out of a {@link org.w3c.dom.Node}
	 * @param node never <code>null</code>
	 * @return the node as string, never <code>null</code>
	 * @throws java.lang.Exception
	 */
	public static String asString(Node node) throws Exception {
		StringWriter writer = new StringWriter();
		Transformer transformer = TransformerFactory.newInstance().newTransformer();
		transformer.transform(new DOMSource(node), new StreamResult(writer));
		return writer.toString();
	}

	/**
	 * <p>
	 * buildQName.
	 * </p>
	 * @return the node of the name as qualified name, never <code>null</code>
	 * @param node a {@link org.w3c.dom.Node} object
	 */
	public static QName buildQName(Node node) {
		String localPart;
		String nsName;
		String name = node.getTextContent();
		int indexOfColon = name.indexOf(':');
		if (indexOfColon > 0) {
			localPart = name.substring(indexOfColon + 1);
			nsName = node.lookupNamespaceURI(name.substring(0, indexOfColon));
		}
		else {
			localPart = name;
			// return default namespace URI if any
			nsName = node.lookupNamespaceURI(null);
		}
		return new QName(nsName, localPart);
	}

	/**
	 * <p>
	 * reloadNode.
	 * </p>
	 * @param wfsCapabilities a {@link org.w3c.dom.Node} object
	 * @return a {@link org.w3c.dom.Node} object
	 * @throws java.lang.Exception if any.
	 * @throws org.xml.sax.SAXException if any.
	 * @throws java.io.IOException if any.
	 * @throws javax.xml.parsers.ParserConfigurationException if any.
	 */
	public static Node reloadNode(Node wfsCapabilities)
			throws Exception, SAXException, IOException, ParserConfigurationException {
		// this is required cause of java.lang.RuntimeException: Knoten konnte
		// nicht in Handle aufgelöst werden
		// at
		// com.sun.org.apache.xml.internal.dtm.ref.DTMManagerDefault.getDTMHandleFromNode(DTMManagerDefault.java:579)
		// at
		// com.sun.org.apache.xpath.internal.XPathContext.getDTMHandleFromNode(XPathContext.java:188)
		// at com.sun.org.apache.xpath.internal.XPath.execute(XPath.java:305)
		// at
		// com.sun.org.apache.xpath.internal.jaxp.XPathImpl.eval(XPathImpl.java:205)
		// at
		// com.sun.org.apache.xpath.internal.jaxp.XPathImpl.evaluate(XPathImpl.java:270)
		String capabilitiesAsString = asString(wfsCapabilities);
		return asNode(capabilitiesAsString);
	}

	/**
	 * Parses all properties of a feature type matching the given type
	 * @param model the parsed schema definition never <code>null</code>
	 * @param featureTypeName the name of the feature type the properties should be parsed
	 * from, never <code>null</code>
	 * @param typeDefs a list of types the properties should match
	 * @return a list of {@link org.apache.xerces.xs.XSElementDeclaration}s part of the
	 * feature type matching to one of the passed types, may be empty but never
	 * <code>null</code>
	 */
	public static List<XSElementDeclaration> getFeaturePropertiesByType(XSModel model, QName featureTypeName,
			XSTypeDefinition... typeDefs) {
		XSElementDeclaration elemDecl = model.getElementDeclaration(featureTypeName.getLocalPart(),
				featureTypeName.getNamespaceURI());
		XSComplexTypeDefinition featureTypeDef = (XSComplexTypeDefinition) elemDecl.getTypeDefinition();
		List<XSElementDeclaration> featureProps = XMLSchemaModelUtils
			.getAllElementsInParticle(featureTypeDef.getParticle());
		List<XSElementDeclaration> props = new ArrayList<XSElementDeclaration>();
		// set bit mask to indicate acceptable derivation mechanisms
		short extendOrRestrict = XSConstants.DERIVATION_EXTENSION | XSConstants.DERIVATION_RESTRICTION;
		for (XSElementDeclaration featureProp : featureProps) {
			XSTypeDefinition propType = featureProp.getTypeDefinition();
			for (XSTypeDefinition typeDef : typeDefs) {
				switch (propType.getTypeCategory()) {
					case XSTypeDefinition.SIMPLE_TYPE:
						if ((typeDef.getTypeCategory() == XSTypeDefinition.SIMPLE_TYPE)
								&& propType.derivedFromType(typeDef, extendOrRestrict)) {
							props.add(featureProp);
						}
						break;
					case XSTypeDefinition.COMPLEX_TYPE:
						if (typeDef.getTypeCategory() == XSTypeDefinition.COMPLEX_TYPE) {
							// check type of child element(s)
							XSComplexTypeDefinition complexPropType = (XSComplexTypeDefinition) propType;
							List<XSElementDeclaration> propValues = XMLSchemaModelUtils
								.getAllElementsInParticle(complexPropType.getParticle());
							for (XSElementDeclaration propValue : propValues) {
								if (propValue.getTypeDefinition().derivedFromType(typeDef, extendOrRestrict)) {
									props.add(featureProp);
								}
							}
						}
						else {
							// complex type may derive from simple type
							if (propType.derivedFromType(typeDef, extendOrRestrict)) {
								props.add(featureProp);
							}
						}
						break;
				}
			}
		}
		return props;
	}

}
