/*
 * Decompiled with CFR 0.152.
 */
package org.opengis.cite.iso19142.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalUnit;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.xml.bind.JAXBException;
import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.stream.StreamSource;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import net.sf.saxon.s9api.SaxonApiException;
import net.sf.saxon.s9api.XdmItem;
import net.sf.saxon.s9api.XdmValue;
import org.apache.xerces.xs.XSComplexTypeDefinition;
import org.apache.xerces.xs.XSElementDeclaration;
import org.apache.xerces.xs.XSModel;
import org.apache.xerces.xs.XSTypeDefinition;
import org.opengis.cite.geomatics.Extents;
import org.opengis.cite.geomatics.gml.GmlUtils;
import org.opengis.cite.geomatics.time.TemporalComparator;
import org.opengis.cite.geomatics.time.TemporalUtils;
import org.opengis.cite.iso19142.FeatureTypeInfo;
import org.opengis.cite.iso19142.ProtocolBinding;
import org.opengis.cite.iso19142.basic.filter.temporal.TemporalQuery;
import org.opengis.cite.iso19142.util.AppSchemaUtils;
import org.opengis.cite.iso19142.util.FeatureProperty;
import org.opengis.cite.iso19142.util.NamespaceBindings;
import org.opengis.cite.iso19142.util.ServiceMetadataUtils;
import org.opengis.cite.iso19142.util.WFSClient;
import org.opengis.cite.iso19142.util.XMLUtils;
import org.opengis.geometry.Envelope;
import org.opengis.temporal.Instant;
import org.opengis.temporal.Period;
import org.opengis.temporal.TemporalGeometricPrimitive;
import org.testng.SkipException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

public class DataSampler {
    private static final Logger LOGR = Logger.getLogger(DataSampler.class.getPackage().getName());
    public static final QName BOUNDED_BY = new QName("http://www.opengis.net/gml/3.2", "boundedBy");
    private int maxFeatures = 25;
    private Document serviceDescription;
    private Map<QName, FeatureTypeInfo> featureInfo;
    private Map<QName, Envelope> spatialExtents;
    private Map<FeatureProperty, Period> temporalPropertyExtents;
    private final Map<QName, List<QName>> nillableProperties = new HashMap<QName, List<QName>>();

    public DataSampler(Document wfsCapabilities) {
        if (null == wfsCapabilities || !wfsCapabilities.getDocumentElement().getLocalName().equals("WFS_Capabilities")) {
            throw new IllegalArgumentException("Did not supply a WFS capabilities document");
        }
        this.serviceDescription = wfsCapabilities;
        this.featureInfo = ServiceMetadataUtils.extractFeatureTypeInfo(wfsCapabilities);
        if (this.featureInfo.isEmpty()) {
            throw new RuntimeException("No feature type info available.");
        }
        this.spatialExtents = new HashMap<QName, Envelope>();
        this.temporalPropertyExtents = new HashMap<FeatureProperty, Period>();
        LOGR.config("Created DataSampler - GetCapabilities (GET) endpoint is " + ServiceMetadataUtils.getOperationEndpoint(wfsCapabilities, "GetCapabilities", ProtocolBinding.GET));
    }

    public void setMaxFeatures(int maxFeatures) {
        if (maxFeatures > 0) {
            this.maxFeatures = maxFeatures;
        }
    }

    public Set<String> selectRandomFeatureIdentifiers(QName featureType, int numId) {
        FeatureTypeInfo typeInfo = this.featureInfo.get(featureType);
        File dataFile = typeInfo.getSampleData();
        HashSet<String> idSet = new HashSet<String>();
        if (null == dataFile || !dataFile.exists()) {
            return idSet;
        }
        String xpath = "//wfs:member/*/@gml:id";
        HashMap<String, String> nsBindings = new HashMap<String, String>();
        nsBindings.put("http://www.opengis.net/gml/3.2", "gml");
        nsBindings.put("http://www.opengis.net/wfs/2.0", "wfs");
        XdmValue result = null;
        try {
            result = XMLUtils.evaluateXPath2(new StreamSource(dataFile), xpath, nsBindings);
        }
        catch (SaxonApiException e) {
            LOGR.log(Level.WARNING, String.format("Failed to extract feature identifiers from data file at %s", dataFile.getAbsolutePath()));
        }
        int sampleSize = result.size();
        numId = numId > sampleSize ? sampleSize : numId;
        HashSet<Integer> randomSet = new HashSet<Integer>();
        Random random = new Random();
        while (randomSet.size() < numId) {
            int randomInt = random.nextInt(sampleSize);
            randomSet.add(randomInt);
        }
        Iterator iterator = randomSet.iterator();
        while (iterator.hasNext()) {
            int randomInt = (Integer)iterator.next();
            String featureIdentifier = result.itemAt(randomInt).getStringValue();
            if (idSet.contains(featureIdentifier)) {
                throw new IllegalArgumentException("Feature id " + featureIdentifier + " exists multiple times in Feature Type " + featureType.toString() + ". This is not allowed according to GML 3.2.1 specification (OGC 07-036; chapter 7.2.4.5).");
            }
            idSet.add(featureIdentifier);
        }
        return idSet;
    }

    public List<String> getSimplePropertyValues(QName featureType, QName propName, String featureId) {
        FeatureTypeInfo typeInfo = this.featureInfo.get(featureType);
        File dataFile = typeInfo.getSampleData();
        ArrayList<String> values = new ArrayList<String>();
        if (null == dataFile || !dataFile.exists()) {
            return values;
        }
        HashMap<String, String> nsBindings = new HashMap<String, String>();
        nsBindings.put("http://www.opengis.net/wfs/2.0", "wfs");
        nsBindings.put("http://www.opengis.net/gml/3.2", "gml");
        nsBindings.put("http://www.w3.org/2001/XMLSchema-instance", "xsi");
        nsBindings.put(featureType.getNamespaceURI(), "ns1");
        StringBuilder xpath = new StringBuilder("//wfs:member/ns1:");
        xpath.append(featureType.getLocalPart());
        if (null != featureId && !featureId.isEmpty()) {
            xpath.append("[@gml:id='").append(featureId).append("']");
        }
        if (!propName.getNamespaceURI().equals(featureType.getNamespaceURI())) {
            xpath.append("/ns2:");
            nsBindings.put(propName.getNamespaceURI(), "ns2");
        } else {
            xpath.append("/ns1:");
        }
        xpath.append(propName.getLocalPart());
        xpath.append("[not(@xsi:nil)]");
        XdmValue result = null;
        try {
            result = XMLUtils.evaluateXPath2(new StreamSource(dataFile), xpath.toString(), nsBindings);
        }
        catch (SaxonApiException e) {
            LOGR.log(Level.WARNING, String.format("Failed to evaluate XPath expression %s against data at %s\n%s\n", xpath, dataFile.getAbsolutePath(), nsBindings) + e.getMessage());
        }
        if (null != result) {
            for (XdmItem item : result) {
                values.add(item.getStringValue());
            }
        }
        if (LOGR.isLoggable(Level.FINE)) {
            LOGR.log(Level.FINE, "[{0}] Evaluating xpath {1}\n {2}", new Object[]{this.getClass().getName(), xpath, values});
        }
        return values;
    }

    public boolean deleteData() {
        boolean allDeleted = true;
        for (QName typeName : this.featureInfo.keySet()) {
            File file = this.featureInfo.get(typeName).getSampleData();
            if (file == null || !file.exists() || file.delete()) continue;
            allDeleted = false;
            LOGR.log(Level.WARNING, "Failed to delete sample data file at " + file);
        }
        return allDeleted;
    }

    public void acquireFeatureData() {
        WFSClient wfsClient = new WFSClient(this.serviceDescription);
        Set<ProtocolBinding> getFeatureBindings = ServiceMetadataUtils.getOperationBindings(this.serviceDescription, "GetFeature");
        if (getFeatureBindings.isEmpty()) {
            throw new IllegalArgumentException("No bindings available for GetFeature request.");
        }
        for (Map.Entry<QName, FeatureTypeInfo> entry : this.featureInfo.entrySet()) {
            QName typeName = entry.getKey();
            FeatureTypeInfo featureTypeInfo = entry.getValue();
            this.acquireFeatureData(wfsClient, getFeatureBindings, typeName, featureTypeInfo);
        }
        LOGR.log(Level.INFO, this.featureInfo.toString());
    }

    public Map<QName, FeatureTypeInfo> getFeatureTypeInfo() {
        return this.featureInfo;
    }

    public Element getFeatureById(String id) {
        XPath xpath = XPathFactory.newInstance().newXPath();
        xpath.setNamespaceContext(NamespaceBindings.withStandardBindings());
        String expr = String.format("//wfs:member/*[@gml:id='%s']", id);
        Element feature = null;
        for (FeatureTypeInfo featureInfo : this.featureInfo.values()) {
            if (!featureInfo.isInstantiated()) continue;
            File dataFile = featureInfo.getSampleData();
            NodeList result = null;
            try {
                result = (NodeList)xpath.evaluate(expr, new InputSource(new FileInputStream(dataFile)), XPathConstants.NODESET);
            }
            catch (FileNotFoundException | XPathExpressionException e) {
                LOGR.log(Level.WARNING, String.format("Failed to evaluate XPath %s against data file at %s", expr, dataFile.getAbsolutePath()));
            }
            if (result.getLength() <= 0) continue;
            feature = (Element)result.item(0);
            break;
        }
        return feature;
    }

    public String getFeatureIdNotOfType(QName featureType) {
        return this.getFeatureId(featureTypeInfo -> featureTypeInfo.getTypeName().equals(featureType));
    }

    public String getFeatureId() {
        return this.getFeatureId(featureTypeInfo -> false);
    }

    private String getFeatureId(Function<FeatureTypeInfo, Boolean> skip) {
        String expr = "(//wfs:member/*/@gml:id)[1]";
        HashMap<String, String> nsBindings = new HashMap<String, String>();
        nsBindings.put("http://www.opengis.net/gml/3.2", "gml");
        nsBindings.put("http://www.opengis.net/wfs/2.0", "wfs");
        for (FeatureTypeInfo featureTypeInfo : this.featureInfo.values()) {
            if (skip.apply(featureTypeInfo).booleanValue() || !featureTypeInfo.isInstantiated()) continue;
            File dataFile = featureTypeInfo.getSampleData();
            try {
                String featureId;
                XdmValue result = XMLUtils.evaluateXPath2(new StreamSource(dataFile), expr, nsBindings);
                if (result.size() <= 0 || (featureId = result.itemAt(0).getStringValue()) == null || featureId.isEmpty()) continue;
                return featureId;
            }
            catch (SaxonApiException e) {
                LOGR.log(Level.WARNING, String.format("Failed to evaluate XPath %s against data file at %s", expr, dataFile.getAbsolutePath()));
            }
        }
        return null;
    }

    public Envelope getSpatialExtent(XSModel model, QName featureType) {
        Envelope envelope = this.spatialExtents.get(featureType);
        if (null != envelope) {
            return envelope;
        }
        List<XSElementDeclaration> geomProps = AppSchemaUtils.getFeaturePropertiesByType(model, featureType, model.getTypeDefinition("AbstractGeometryType", "http://www.opengis.net/gml/3.2"));
        if (geomProps.isEmpty()) {
            return null;
        }
        Iterator<XSElementDeclaration> itr = geomProps.iterator();
        NamespaceBindings nsBindings = NamespaceBindings.withStandardBindings();
        XPathFactory factory = XPathFactory.newInstance();
        NodeList geomNodes = null;
        File dataFile = this.featureInfo.get(featureType).getSampleData();
        do {
            XSElementDeclaration geomProp = itr.next();
            nsBindings.addNamespaceBinding(geomProp.getNamespace(), "ns1");
            String expr = String.format("//ns1:%s/*[1]", geomProp.getName());
            XPath xpath = factory.newXPath();
            xpath.setNamespaceContext(nsBindings);
            try {
                geomNodes = (NodeList)xpath.evaluate(expr, new InputSource(new FileInputStream(dataFile)), XPathConstants.NODESET);
            }
            catch (FileNotFoundException | XPathExpressionException e) {
                LOGR.log(Level.WARNING, String.format("Failed to evaluate XPath %s against data file at %s.\n %s", expr, dataFile.getAbsolutePath(), e.getMessage()));
            }
        } while ((null == geomNodes || geomNodes.getLength() <= 0) && itr.hasNext());
        if (null != geomNodes && geomNodes.getLength() > 0) {
            try {
                envelope = Extents.calculateEnvelopeUsingSingleGeometry((NodeList)geomNodes);
            }
            catch (JAXBException e) {
                LOGR.log(Level.WARNING, String.format("Failed to create envelope from geometry nodes.", e.getMessage()));
            }
        }
        this.spatialExtents.put(featureType, envelope);
        return envelope;
    }

    public Period getTemporalExtentOfProperty(XSModel model, QName featureType, XSElementDeclaration tmPropDecl) {
        Document data;
        FeatureProperty tmProp;
        try {
            tmProp = new FeatureProperty(featureType, tmPropDecl);
        }
        catch (Exception e) {
            throw new IllegalArgumentException("Property " + tmPropDecl + " is not suitable as temporal property.", e);
        }
        Period period = this.temporalPropertyExtents.get(tmProp);
        if (null != period) {
            return period;
        }
        File dataFile = this.featureInfo.get(featureType).getSampleData();
        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setNamespaceAware(true);
            data = factory.newDocumentBuilder().parse(dataFile);
        }
        catch (IOException | ParserConfigurationException | SAXException e) {
            throw new RuntimeException(String.format("Failed to parse data file at %s.\n %s", dataFile.getAbsolutePath(), e.getMessage()));
        }
        TreeSet<TemporalGeometricPrimitive> tmSet = new TreeSet<TemporalGeometricPrimitive>((Comparator<TemporalGeometricPrimitive>)new TemporalComparator());
        NodeList propNodes = data.getElementsByTagNameNS(tmPropDecl.getNamespace(), tmPropDecl.getName());
        for (int i = 0; i < propNodes.getLength(); ++i) {
            XSTypeDefinition propType = tmPropDecl.getTypeDefinition();
            Element propElem = (Element)propNodes.item(i);
            try {
                TemporalGeometricPrimitive tVal;
                if (propType.getTypeCategory() == 16 || ((XSComplexTypeDefinition)propType).getContentType() == 1) {
                    tVal = TemporalQuery.parseTemporalValue(propElem.getTextContent(), propType);
                } else {
                    Element propValue = (Element)propElem.getElementsByTagName("*").item(0);
                    tVal = GmlUtils.gmlToTemporalGeometricPrimitive((Element)propValue);
                }
                tmSet.add(tVal);
                continue;
            }
            catch (RuntimeException re) {
                LOGR.log(Level.WARNING, re.getMessage());
                if (!(re instanceof SkipException)) continue;
                String message = re.getMessage();
                message = String.format(message, featureType.toString());
                throw new SkipException(message);
            }
        }
        if (period != null) {
            TemporalUtils.add((Instant)period.getEnding(), (int)2, (TemporalUnit)ChronoUnit.DAYS);
            TemporalUtils.add((Instant)period.getBeginning(), (int)-2, (TemporalUnit)ChronoUnit.DAYS);
        }
        period = TemporalUtils.temporalExtent(tmSet);
        this.temporalPropertyExtents.put(tmProp, period);
        return period;
    }

    public List<QName> getNillableProperties(XSModel model, QName featureType) {
        if (this.nillableProperties.containsKey(featureType)) {
            return this.nillableProperties.get(featureType);
        }
        ArrayList<QName> nillableProperties = new ArrayList<QName>();
        LOGR.fine("Checking feature type for nillable properties: " + featureType);
        List<XSElementDeclaration> nillableProps = AppSchemaUtils.getNillableProperties(model, featureType);
        LOGR.fine(nillableProps.toString());
        FeatureTypeInfo typeInfo = this.getFeatureTypeInfo().get(featureType);
        if (typeInfo.isInstantiated()) {
            for (XSElementDeclaration elementDeclaration : nillableProps) {
                QName propName = new QName(elementDeclaration.getNamespace(), elementDeclaration.getName());
                if (BOUNDED_BY.equals(propName) || !this.nillablePropertyContainsNilledProperties(typeInfo, propName)) continue;
                nillableProperties.add(propName);
            }
        }
        LOGR.fine("Nillable properties:\n" + nillableProps);
        this.nillableProperties.put(featureType, nillableProperties);
        return nillableProperties;
    }

    private boolean nillablePropertyContainsNilledProperties(FeatureTypeInfo typeInfo, QName propertyName) {
        Document data;
        LOGR.fine("Checking property " + propertyName + " for nilled properties.");
        File dataFile = typeInfo.getSampleData();
        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setNamespaceAware(true);
            data = factory.newDocumentBuilder().parse(dataFile);
        }
        catch (IOException | ParserConfigurationException | SAXException e) {
            throw new RuntimeException(String.format("Failed to parse data file at %s.\n %s", dataFile.getAbsolutePath(), e.getMessage()));
        }
        NodeList propNodes = data.getElementsByTagNameNS(propertyName.getNamespaceURI(), propertyName.getLocalPart());
        for (int i = 0; i < propNodes.getLength(); ++i) {
            Element propElem = (Element)propNodes.item(i);
            String nilValue = propElem.getAttributeNS("http://www.w3.org/2001/XMLSchema-instance", "nil");
            if (!"true".equals(nilValue)) continue;
            return true;
        }
        LOGR.fine("Property " + propertyName + " does not have nilled properties.");
        return false;
    }

    public Element randomlySelectFeatureInstance() {
        Element feature = null;
        for (Map.Entry<QName, FeatureTypeInfo> entry : this.featureInfo.entrySet()) {
            if (!entry.getValue().isInstantiated()) continue;
            Set<String> idSet = this.selectRandomFeatureIdentifiers(entry.getKey(), 1);
            feature = this.getFeatureById(idSet.iterator().next());
        }
        return feature;
    }

    public QName selectFeatureType() {
        ArrayList<FeatureTypeInfo> availableTypes = new ArrayList<FeatureTypeInfo>();
        ArrayList<String> featureName = new ArrayList<String>();
        for (FeatureTypeInfo typeInfo : this.featureInfo.values()) {
            if (!typeInfo.isInstantiated()) continue;
            availableTypes.add(typeInfo);
            featureName.add(typeInfo.getTypeName().getLocalPart());
        }
        if (availableTypes.isEmpty()) {
            return null;
        }
        Collections.sort(featureName);
        Optional<FeatureTypeInfo> availableType = availableTypes.stream().filter(x -> x.getTypeName().getLocalPart().equalsIgnoreCase((String)featureName.get(0))).findFirst();
        return availableType.get().getTypeName();
    }

    public XdmValue evaluateXPathAgainstSampleData(String expr, Map<String, String> nsBindings) {
        XdmValue results = null;
        for (Map.Entry<QName, FeatureTypeInfo> entry : this.featureInfo.entrySet()) {
            if (!entry.getValue().isInstantiated()) continue;
            File dataFile = entry.getValue().getSampleData();
            try {
                results = XMLUtils.evaluateXPath2(new StreamSource(dataFile), expr, nsBindings);
                if (results.size() <= 0) continue;
                break;
            }
            catch (SaxonApiException e) {
                LOGR.log(Level.WARNING, e.getMessage());
            }
        }
        return results;
    }

    private void acquireFeatureData(WFSClient wfsClient, Set<ProtocolBinding> getFeatureBindings, QName typeName, FeatureTypeInfo featureTypeInfo) {
        for (ProtocolBinding binding : getFeatureBindings) {
            try {
                Document rspEntity = wfsClient.getFeatureByType(typeName, this.maxFeatures, binding);
                NodeList features = rspEntity.getElementsByTagNameNS(typeName.getNamespaceURI(), typeName.getLocalPart());
                boolean hasFeatures = features.getLength() > 0;
                if (!hasFeatures) continue;
                this.saveFeatureDataFile(featureTypeInfo, typeName, rspEntity);
                return;
            }
            catch (RuntimeException re) {
                StringBuilder err = new StringBuilder();
                err.append(String.format("Failed to read XML response entity using %s method for feature type %s.", new Object[]{binding, typeName}));
                err.append(" \n").append(re.getMessage());
                LOGR.log(Level.WARNING, err.toString(), re);
            }
        }
    }

    private void saveFeatureDataFile(FeatureTypeInfo featureTypeInfo, QName typeName, Document rspEntity) {
        try {
            File file = File.createTempFile(typeName.getLocalPart() + "-", ".xml");
            FileOutputStream fos = new FileOutputStream(file);
            XMLUtils.writeNode(rspEntity, fos);
            LOGR.log(Level.CONFIG, this.getClass().getName() + " - wrote feature data to " + file.getAbsolutePath());
            featureTypeInfo.setSampleData(file);
            fos.close();
            featureTypeInfo.setInstantiated(true);
        }
        catch (Exception e) {
            LOGR.log(Level.WARNING, "Failed to save feature data.", e);
        }
    }
}

