/*
 * Decompiled with CFR 0.152.
 */
package org.openksavi.sponge.midi;

import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import javax.sound.midi.Instrument;
import javax.sound.midi.InvalidMidiDataException;
import javax.sound.midi.MidiChannel;
import javax.sound.midi.MidiDevice;
import javax.sound.midi.MidiDeviceTransmitter;
import javax.sound.midi.MidiSystem;
import javax.sound.midi.MidiUnavailableException;
import javax.sound.midi.Receiver;
import javax.sound.midi.Sequence;
import javax.sound.midi.Sequencer;
import javax.sound.midi.ShortMessage;
import javax.sound.midi.Soundbank;
import javax.sound.midi.Synthesizer;
import javax.sound.midi.Track;
import javax.sound.midi.Transmitter;
import org.openksavi.sponge.SpongeException;
import org.openksavi.sponge.core.util.SpongeUtils;
import org.openksavi.sponge.midi.MidiPlugin;
import org.openksavi.sponge.midi.event.MidiShortMessageEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class MidiUtils {
    private static final Logger logger = LoggerFactory.getLogger(MidiUtils.class);
    private static final String[] NOTE_NAMES = new String[]{"C", "C#", "D", "D#", "E", "F", "F#", "G", "G#", "A", "A#", "B"};

    public static List<MidiDevice> getDevices() {
        return Arrays.stream(MidiSystem.getMidiDeviceInfo()).map(info -> {
            try {
                return MidiSystem.getMidiDevice(info);
            }
            catch (MidiUnavailableException e) {
                throw SpongeUtils.wrapException((Throwable)e);
            }
        }).collect(Collectors.toList());
    }

    public static MidiDevice getDefaultInputDevice() {
        try {
            Transmitter transmitter = MidiSystem.getTransmitter();
            if (transmitter != null && transmitter instanceof MidiDeviceTransmitter) {
                return ((MidiDeviceTransmitter)transmitter).getMidiDevice();
            }
            return null;
        }
        catch (MidiUnavailableException e) {
            throw SpongeUtils.wrapException((Throwable)e);
        }
    }

    public static MidiDevice getDeviceByName(String name) {
        return MidiUtils.getDevice(device -> device.getDeviceInfo().getName().equals(name));
    }

    public static MidiDevice getDevice(Predicate<? super MidiDevice> predicate) {
        Optional<? super MidiDevice> result = MidiUtils.getDevices().stream().filter(predicate).findFirst();
        return result.isPresent() ? result.get() : null;
    }

    public static Map<MidiDevice.Info, MidiDevice> getDeviceMap() {
        LinkedHashMap<MidiDevice.Info, MidiDevice> result = new LinkedHashMap<MidiDevice.Info, MidiDevice>();
        try {
            for (MidiDevice.Info info : MidiSystem.getMidiDeviceInfo()) {
                result.put(info, MidiSystem.getMidiDevice(info));
            }
        }
        catch (MidiUnavailableException e) {
            throw SpongeUtils.wrapException((Throwable)e);
        }
        return result;
    }

    public static void open(MidiDevice device) {
        if (device != null && !device.isOpen()) {
            try {
                device.open();
            }
            catch (MidiUnavailableException e) {
                throw SpongeUtils.wrapException((Throwable)e);
            }
        }
    }

    public static void close(MidiDevice device) {
        try {
            if (device != null && device.isOpen()) {
                device.close();
            }
        }
        catch (Exception e) {
            logger.warn("close", (Throwable)e);
        }
    }

    public static void close(Transmitter transmitter) {
        try {
            if (transmitter != null) {
                transmitter.close();
            }
        }
        catch (Exception e) {
            logger.warn("close", (Throwable)e);
        }
    }

    public static void close(Receiver receiver) {
        try {
            if (receiver != null) {
                receiver.close();
            }
        }
        catch (Exception e) {
            logger.warn("close", (Throwable)e);
        }
    }

    public static ShortMessage createShortMessage(int command, int channel, int data1, int data2) {
        ShortMessage result = new ShortMessage();
        try {
            result.setMessage(command, channel, data1, data2);
        }
        catch (InvalidMidiDataException e) {
            throw SpongeUtils.wrapException((Throwable)e);
        }
        return result;
    }

    public static void setMute(Sequencer sequencer, Sequence sequence, boolean mute) {
        Track[] tracks = sequence.getTracks();
        for (int i = 0; i < tracks.length; ++i) {
            sequencer.setTrackMute(i, mute);
        }
    }

    public static MidiShortMessageEvent createShortMessageEvent(MidiPlugin midiPlugin, ShortMessage message, long timeStamp) {
        return new MidiShortMessageEvent(midiPlugin.getMidiShortMessageEventName(), midiPlugin.getEngine().getDefaultParameters().getEventClonePolicy(), message, (Long)timeStamp);
    }

    public static MidiShortMessageEvent createShortMessageEvent(MidiPlugin midiPlugin, ShortMessage message) {
        return MidiUtils.createShortMessageEvent(midiPlugin, message, -1L);
    }

    public static String getKeyNote(int key) {
        if (key < 0 || key > 127) {
            throw new IllegalArgumentException("MIDI key is " + key + " but should be a value between 0 and 127");
        }
        return NOTE_NAMES[key % 12] + key / 12;
    }

    public static Instrument loadInstrument(Synthesizer synthesizer, String instrumentName) {
        Optional<Instrument> result = Arrays.stream(synthesizer.getAvailableInstruments()).filter(instrument -> instrument.getName().equals(instrumentName)).findFirst();
        if (!result.isPresent()) {
            throw new IllegalArgumentException("Instrument named " + instrumentName + " is not available");
        }
        return synthesizer.loadInstrument(result.get()) ? result.get() : null;
    }

    public static MidiChannel getChannel(Synthesizer synthesizer, int channel, boolean required) {
        MidiChannel[] channels = synthesizer.getChannels();
        if (channel >= channels.length || channels[channel] == null) {
            if (required) {
                throw new IllegalArgumentException("Channel " + channel + " is not available");
            }
            return null;
        }
        return channels[channel];
    }

    public static Instrument getInstrument(Synthesizer synthesizer, int channel) {
        return synthesizer.getAvailableInstruments()[MidiUtils.getChannel(synthesizer, channel, true).getProgram()];
    }

    public static Instrument setInstrument(Synthesizer synthesizer, int channel, int instrument) {
        MidiUtils.getChannel(synthesizer, channel, true).programChange(instrument);
        return MidiUtils.getInstrument(synthesizer, channel);
    }

    public static Instrument setInstrument(Synthesizer synthesizer, int channel, String instrumentName) {
        Optional<Instrument> result = Arrays.stream(synthesizer.getAvailableInstruments()).filter(instrument -> instrument.getName().equals(instrumentName)).findFirst();
        if (!result.isPresent()) {
            throw new IllegalArgumentException("Instrument named " + instrumentName + " is not available");
        }
        return MidiUtils.setInstrument(synthesizer, channel, result.get().getPatch().getProgram());
    }

    public static boolean loadAllInstruments(Synthesizer synthesizer, Soundbank soundbank) {
        return synthesizer.loadAllInstruments(soundbank);
    }

    public static boolean loadAllInstruments(Synthesizer synthesizer) {
        if (synthesizer.getDefaultSoundbank() != null) {
            return MidiUtils.loadAllInstruments(synthesizer, synthesizer.getDefaultSoundbank());
        }
        return false;
    }

    public static List<Instrument> getInstruments(Synthesizer synthesizer) {
        return Arrays.stream(synthesizer.getLoadedInstruments()).collect(Collectors.toList());
    }

    public static void closeTransmitters(MidiDevice device) {
        device.getTransmitters().forEach(Transmitter::close);
    }

    public static int getFirstSupportedMidiFileType(Sequence sequence) {
        int[] fileTypes = MidiSystem.getMidiFileTypes(sequence);
        if (fileTypes.length == 0) {
            throw new SpongeException("No supported MIDI file format");
        }
        return fileTypes[0];
    }

    private MidiUtils() {
    }
}

