package org.openlr.geo;

import org.locationtech.jts.geom.*;
import org.locationtech.jts.util.GeometricShapeFactory;

import java.util.ArrayList;
import java.util.List;

/**
 * A collection of geometric operating on geometry with coordinates in the WGS84 coordinate system.
 */
public class Geo {
    private static final int EARTH_RADIUS_METERS = 6371000;
    private static final int EARTH_CIRCUMFERENCE_METERS = 40075000;
    private static final int DEGREES_IN_CIRCLE = 360;

    private final GeometryFactory geometryFactory;

    public Geo(GeometryFactory geometryFactory) {
        this.geometryFactory = geometryFactory;
    }

    public Geo() {
        this(new GeometryFactory());
    }

    /**
     * Calculate the distance in meters between a pair of coordinates.
     *
     * @param coordinateA the first coordinate
     * @param coordinateB the second coordinate
     * @return the distance in meters between the coordinate pair
     */
    public double calculateDistance(Coordinate coordinateA, Coordinate coordinateB) {
        double aYRadians = Math.toRadians(coordinateA.y);
        double bYRadians = Math.toRadians(coordinateB.y);

        double dX = Math.toRadians(coordinateB.x - coordinateA.x);
        double dY = Math.toRadians(coordinateB.y - coordinateA.y);

        double a = (Math.cos(aYRadians) * Math.cos(bYRadians) * Math.sin(dX / 2) * Math.sin(dX / 2)) +
                (Math.sin(dY / 2) * Math.sin(dY / 2));

        double c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));

        return c * EARTH_RADIUS_METERS;
    }

    /**
     * Calculate the distance in meters between a pair of points.
     *
     * @param pointA the first point
     * @param pointB the second point
     * @return the distance in meters between the point pair
     */
    public double calculateDistance(Point pointA, Point pointB) {
        return calculateDistance(pointA.getCoordinate(), pointB.getCoordinate());
    }

    /**
     * Calculate the bearing angle between a pair of coordinates. The angle is returned in
     * degrees {@code 0° >= angle < 360°} and is the angle in a clockwise direction relative to
     * due north.
     *
     * @param from the origin coordinate
     * @param to the destination coordinate
     * @return the bearing angle from origin to destination
     */
    public double calculateBearing(Coordinate from, Coordinate to) {
        double fromXRadians = Math.toRadians(from.x);
        double fromYRadians = Math.toRadians(from.y);

        double toXRadians = Math.toRadians(to.x);
        double toYRadians = Math.toRadians(to.y);

        double y = Math.sin(toXRadians - fromXRadians) * Math.cos(toYRadians);
        double x = (Math.cos(fromYRadians) * Math.sin(toYRadians)) -
                (Math.sin(fromYRadians) * Math.cos(toYRadians) * Math.cos(toXRadians - fromXRadians));

        double theta = Math.atan2(y, x);

        return (Math.toDegrees(theta) + 360.0) % 360.0;
    }

    /**
     * Calculate the bearing angle between a pair of points. The angle is returned in
     * degrees {@code 0° >= angle < 360°} and is the angle in a clockwise direction relative to
     * due north.
     *
     * @param from the origin point
     * @param to the destination point
     * @return the bearing angle from origin to destination
     */
    public double calculateBearing(Point from, Point to) {
        return calculateBearing(from.getCoordinate(), to.getCoordinate());
    }

    /**
     * Calculate the absolute angle difference in degrees between a pair of bearing angles.
     * The angle is returned in degrees {@code 0° >= angle < 180°}.
     *
     * @param bearingA the first bearing angle
     * @param bearingB the second bearing angle
     * @return the difference between the pair of angles
     */
    public double calculateBearingDifference(double bearingA, double bearingB) {
        double difference = Math.abs(bearingB - bearingA) % DEGREES_IN_CIRCLE;

        return difference > (double) DEGREES_IN_CIRCLE / 2 ?
                DEGREES_IN_CIRCLE - difference :
                difference;
    }

    /**
     * Project a coordinate onto a line string and return fraction along the line string
     * from the start where the projection point is found.
     *
     * @param coordinate the coordinate to project
     * @param lineString the line string to project onto
     * @return the fraction along the line string where the projection point is found
     */
    public double projectOnLineString(Coordinate coordinate, LineString lineString) {
        double totalLength = 0;
        Double minDistance = null;
        Double distanceAlong = null;

        for (int i = 1; i < lineString.getNumPoints(); i++) {
            Coordinate from = lineString.getCoordinateN(i - 1);
            Coordinate to = lineString.getCoordinateN(i);

            double segmentLength = calculateDistance(from, to);

            double fraction = project(coordinate, from, to);
            Coordinate projectionPoint = coordinateAlong(from, to, fraction);

            double projectionDistance = calculateDistance(coordinate, projectionPoint);

            if (minDistance == null || projectionDistance < minDistance) {
                minDistance = projectionDistance;
                distanceAlong = totalLength + (segmentLength * fraction);
            }

            totalLength += segmentLength;
        }

        if (distanceAlong == null) {
            throw new IllegalStateException();
        }

        return distanceAlong / totalLength;
    }

    /**
     * Calculate the coordinate along a line string at a certain fraction of the line string's
     * length.
     *
     * @param lineString the line string
     * @param fraction the fraction along the line string
     * @return the coordinate at this fraction
     */
    public Coordinate calculateCoordinateAlongLineString(LineString lineString, double fraction) {
        if (fraction < 0.0 || fraction > 1.0) {
            throw new IllegalArgumentException("Fraction should be between 0 and 1");
        }

        if (fraction == 0) {
            return lineString.getCoordinateN(0);
        }
        else if (fraction == 1.0) {
            return lineString.getCoordinateN(lineString.getNumPoints() - 1);
        }
        else if (lineString.getNumPoints() == 2) {
            Coordinate from = lineString.getCoordinateN(0);
            Coordinate to = lineString.getCoordinateN(1);

            return coordinateAlong(from, to, fraction);
        }
        else {
            List<Double> segmentLengths = new ArrayList<>();
            double totalLength = 0;

            for (int i = 1; i < lineString.getNumPoints(); i++) {
                Coordinate from = lineString.getCoordinateN(i - 1);
                Coordinate to = lineString.getCoordinateN(i);

                double segmentLength = calculateDistance(from, to);
                segmentLengths.add(segmentLength);

                totalLength += segmentLength;
            }

            double distanceAlong = totalLength * fraction;
            int i = 0;

            while (distanceAlong > segmentLengths.get(i)) {
                distanceAlong -= segmentLengths.get(i);
                i++;
            }

            double fractionAlongSegment = distanceAlong / segmentLengths.get(i);

            Coordinate from = lineString.getCoordinateN(i);
            Coordinate to = lineString.getCoordinateN(i + 1);

            return coordinateAlong(from, to, fractionAlongSegment);
        }
    }

    /**
     * Calculate the point along a line string at a certain fraction of the line string's
     * length.
     *
     * @param lineString the line string
     * @param fraction the fraction along the line string
     * @return the point at this fraction
     */
    public Point calculatePointAlongLineString(LineString lineString, double fraction) {
        Coordinate coordinate = calculateCoordinateAlongLineString(lineString, fraction);
        return geometryFactory.createPoint(coordinate);
    }

    /**
     * Join a sequence of line strings together to form a single line string. The line strings
     * must be connected ie. the start point of any line string must be the same as the end point
     * of the preceding line string.
     *
     * @param lineStrings a sequence of line strings
     * @return the concatenation of the line strings
     */
    public LineString joinLineStrings(List<LineString> lineStrings) {
        if (lineStrings.isEmpty()) {
            throw new IllegalArgumentException("At least one line string expected");
        }

        LineString firstLineString = lineStrings.get(0);
        Coordinate currentCoordinate = firstLineString.getCoordinateN(0);

        List<Coordinate> coordinates = new ArrayList<>();
        coordinates.add(currentCoordinate);

        for (LineString lineString : lineStrings) {
            Coordinate firstCoordinate = lineString.getCoordinateN(0);

            if (!firstCoordinate.equals(currentCoordinate)) {
                throw new IllegalArgumentException("Line strings are not connected");
            }

            for (int i = 1; i < lineString.getNumPoints(); i++) {
                currentCoordinate = lineString.getCoordinateN(i);
                coordinates.add(currentCoordinate);
            }
        }

        return geometryFactory.createLineString(coordinates.toArray(Coordinate[]::new));
    }

    /**
     * Create a slice of a line string that begins at a start fraction and ends at a stop fraction
     * of the line string's length.
     *
     * @param lineString the line string to slice
     * @param startFraction the fraction along the line string where the slice begins
     * @param stopFraction the fraction along the line string where the slice ends
     * @return the slice of the line string
     */
    public LineString sliceLineString(LineString lineString, double startFraction, double stopFraction) {
        List<Double> segmentLengths = new ArrayList<>();
        double totalLength = 0;

        for (int i = 1; i < lineString.getNumPoints(); i++) {
            Coordinate from = lineString.getCoordinateN(i - 1);
            Coordinate to = lineString.getCoordinateN(i);

            double segmentLength = calculateDistance(from, to);
            segmentLengths.add(segmentLength);

            totalLength += segmentLength;
        }

        double startLength = totalLength * startFraction;
        double stopLength = totalLength * stopFraction;

        double lengthSoFar = 0;
        List<Coordinate> coordinates = new ArrayList<>();

        for (int i = 1; i < lineString.getNumPoints(); i++) {
            double segmentLength = segmentLengths.get(i - 1);
            double nextLengthSoFar = lengthSoFar + segmentLength;

            Coordinate from = lineString.getCoordinateN(i - 1);
            Coordinate to = lineString.getCoordinateN(i);

            if (nextLengthSoFar >= startLength && coordinates.isEmpty()) {
                double overshoot = nextLengthSoFar - startLength;
                double fractionAlong = (segmentLength - overshoot) / segmentLength;
                Coordinate start = coordinateAlong(from, to, fractionAlong);
                coordinates.add(start);
            }

            if (lengthSoFar > startLength && lengthSoFar < stopLength) {
                coordinates.add(from);
            }

            if (nextLengthSoFar >= stopLength) {
                double overshoot = nextLengthSoFar - stopLength;
                double fractionAlong = (segmentLength - overshoot) / segmentLength;
                Coordinate stop = coordinateAlong(from, to, fractionAlong);
                coordinates.add(stop);
                break;
            }

            lengthSoFar = nextLengthSoFar;
        }

        return geometryFactory.createLineString(coordinates.toArray(Coordinate[]::new));
    }

    /**
     * Create a polygon geometry that approximates a circle.
     *
     * @param center the center point of the circle
     * @param radius the radius of the circle in meters
     * @return a polygon that approximates the circle geometry
     */
    public Polygon approximateCircle(Coordinate center, double radius) {
        return approximateCircle(center, radius, 32);
    }

    /**
     * Create a polygon geometry that approximates a circle. The number of points in the resulting
     * polygon shell is provided. This controls the accuracy of the approximation.
     *
     * @param center the center point of the circle
     * @param radius the radius of the circle in meters
     * @param numberOfPoints the number of points in the polygon shell
     * @return a polygon that approximates the circle geometry
     */
    public Polygon approximateCircle(Coordinate center, double radius, int numberOfPoints) {
        double width = (radius * 2) / (EARTH_CIRCUMFERENCE_METERS * Math.cos(Math.toRadians(center.getY())) / DEGREES_IN_CIRCLE);
        double height = (radius * 2) / (double) (EARTH_CIRCUMFERENCE_METERS / DEGREES_IN_CIRCLE);

        GeometricShapeFactory geometricShapeFactory = new GeometricShapeFactory();
        geometricShapeFactory.setCentre(center);
        geometricShapeFactory.setWidth(width);
        geometricShapeFactory.setHeight(height);
        geometricShapeFactory.setNumPoints(numberOfPoints);

        return geometricShapeFactory.createEllipse();
    }

    private double project(Coordinate coordinate, Coordinate coordinateA, Coordinate coordinateB) {
        if (coordinate.equals(coordinateA)) {
            return 0;
        }
        else if (coordinate.equals(coordinateB)) {
            return 1;
        }
        else {
            double d1 = calculateDistance(coordinateA, coordinateB);
            double d2 = calculateDistance(coordinateA, coordinate);
            double d3 = calculateDistance(coordinateB, coordinate);

            double d1Square = d1 * d1;
            double d2Square = d2 * d2;
            double d3Square = d3 * d3;

            double cos = (d2Square + d1Square - d3Square) / (2.0 * d2 * d1);
            double d = cos * d2;

            double fraction = d / d1;

            return Math.min(Math.max(fraction, 0.0), 1.0);
        }
    }

    private Coordinate coordinateAlong(Coordinate from, Coordinate to, double fraction) {
        double x = from.x + ((to.x - from.x) * fraction);
        double y = from.y + ((to.y - from.y) * fraction);

        return new Coordinate(x, y);
    }
}
