/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.alec.driver.test;

import java.io.File;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import org.opennms.alec.datasource.api.Alarm;
import org.opennms.alec.datasource.api.AlarmDatasource;
import org.opennms.alec.datasource.api.AlarmFeedback;
import org.opennms.alec.datasource.api.InventoryDatasource;
import org.opennms.alec.datasource.api.InventoryObject;
import org.opennms.alec.datasource.api.Situation;
import org.opennms.alec.datasource.api.SituationDatasource;
import org.opennms.alec.datasource.api.SituationHandler;
import org.opennms.alec.engine.api.Engine;
import org.opennms.alec.engine.api.EngineFactory;
import org.opennms.alec.features.graph.api.GraphProvider;
import org.opennms.alec.features.graph.common.GraphMLConverterBuilder;
import org.opennms.alec.features.graph.graphml.GraphML;
import org.opennms.alec.features.graph.graphml.GraphMLWriter;

public class TestDriver {
    private final EngineFactory engineFactory;
    private boolean verbose = false;
    private long graphExportIntervalMs = 0L;
    private File graphOutputFolder = null;

    public TestDriver(EngineFactory engineFactory) {
        this.engineFactory = Objects.requireNonNull(engineFactory);
    }

    public List<Situation> run(List<Alarm> alarms) {
        return this.run(alarms, Collections.emptyList());
    }

    public List<Situation> run(List<Alarm> alarms, List<InventoryObject> inventory, List<Situation> previousSituations, long timestamp) {
        List<Alarm> sortedAlarms = TestDriver.timeSortAlarms(alarms);
        List<Alarm> previousAlarms = TestDriver.getStartupAlarms(sortedAlarms, timestamp);
        List<Alarm> afterAlarms = sortedAlarms.stream().filter(a -> a.getTime() > timestamp).collect(Collectors.toList());
        return this.run(afterAlarms, inventory, previousAlarms, Collections.emptyList(), previousSituations);
    }

    public static List<Alarm> getStartupAlarms(List<Alarm> sortedAlarms, long timestamp) {
        return TestDriver.reduceAlarms(sortedAlarms.stream().filter(a -> a.getTime() <= timestamp).collect(Collectors.toList()));
    }

    public static List<Alarm> reduceAlarms(List<Alarm> sortedAlarms) {
        HashMap<String, Alarm> reduced = new HashMap<String, Alarm>();
        for (Alarm a2 : sortedAlarms) {
            reduced.put(a2.getId(), a2);
        }
        return TestDriver.timeSortAlarms(reduced.values().stream().filter(a -> !a.isClear()).collect(Collectors.toList()));
    }

    public static List<Alarm> timeSortAlarms(List<Alarm> alarms) {
        return alarms.stream().sorted(Comparator.comparing(Alarm::getTime).thenComparing(Alarm::getId)).collect(Collectors.toList());
    }

    public List<Situation> run(List<Alarm> alarms, List<InventoryObject> inventory) {
        return this.run(alarms, inventory, Collections.emptyList(), Collections.emptyList(), Collections.emptyList());
    }

    private long roundToTick(long time, long tickResolutionMs) {
        return Math.floorDiv(time, tickResolutionMs) * tickResolutionMs;
    }

    private List<Situation> run(List<Alarm> alarms, List<InventoryObject> inventory, List<Alarm> previousAlarms, List<AlarmFeedback> previousAlarmFeedback, List<Situation> previousSituations) {
        long now;
        DriverSession session = new DriverSession();
        Engine engine = this.engineFactory.createEngine();
        engine.registerSituationHandler((SituationHandler)session);
        engine.init(previousAlarms, previousAlarmFeedback, previousSituations, inventory);
        GraphProvider graphProvider = engine instanceof GraphProvider ? (GraphProvider)engine : null;
        boolean shouldExportGraph = this.graphExportIntervalMs > 0L && graphProvider != null && this.graphOutputFolder != null;
        long lastGraphGeneratedAt = 0L;
        long tickResolutionMs = engine.getTickResolutionMs();
        Map<Long, List<Alarm>> alarmsByTick = alarms.stream().collect(Collectors.groupingBy(a -> this.roundToTick(a.getTime(), tickResolutionMs)));
        if (alarmsByTick.size() < 1) {
            return Collections.emptyList();
        }
        long start = Math.max(alarms.stream().min(Comparator.comparing(Alarm::getTime)).map(e -> this.roundToTick(e.getTime(), tickResolutionMs)).get() - tickResolutionMs, 0L);
        long end = alarms.stream().max(Comparator.comparing(Alarm::getTime)).map(e -> this.roundToTick(e.getTime(), tickResolutionMs)).get() + tickResolutionMs;
        long startTime = System.currentTimeMillis();
        for (now = start; now <= end; now += tickResolutionMs) {
            for (Alarm alarm : alarmsByTick.getOrDefault(now, Collections.emptyList())) {
                if (!alarm.isClear()) {
                    engine.onAlarmCreatedOrUpdated(alarm);
                    continue;
                }
                engine.onAlarmCleared(alarm);
            }
            engine.tick(now);
            if (shouldExportGraph && now - lastGraphGeneratedAt > this.graphExportIntervalMs) {
                this.exportGraph(graphProvider, now);
                lastGraphGeneratedAt = now;
            }
            this.printTick(now, start, end, startTime);
        }
        engine.tick(now += tickResolutionMs);
        if (shouldExportGraph) {
            this.exportGraph(graphProvider, now);
        }
        engine.destroy();
        return new ArrayList<Situation>(session.situations.values());
    }

    private void exportGraph(GraphProvider graphProvider, long now) {
        if (graphProvider == null) {
            return;
        }
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("MMM_dd_yyyy_HH_mm_ss");
        String dateFormatted = formatter.format(LocalDateTime.ofInstant(Instant.ofEpochMilli(now), ZoneOffset.systemDefault()));
        File destinationFile = new File(this.graphOutputFolder, "graph_" + dateFormatted + ".graphml");
        GraphML graphML = (GraphML)graphProvider.withReadOnlyGraph(g -> new GraphMLConverterBuilder().withGraph(g).withFilterEnptyNodes(false).build().toGraphML());
        try {
            System.out.printf("Saving graph snapshot to %s\n", destinationFile);
            GraphMLWriter.write((GraphML)graphML, (File)destinationFile, (GraphMLWriter.ProcessHook[])new GraphMLWriter.ProcessHook[0]);
        }
        catch (Exception e) {
            System.out.printf("Failed to save graph to: %s\n", destinationFile);
            e.printStackTrace();
        }
    }

    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    private void setGraphExportIntervalMs(long graphExportIntervalMs) {
        this.graphExportIntervalMs = graphExportIntervalMs;
    }

    private void setGraphOutputFolder(File graphOutputFolder) {
        this.graphOutputFolder = graphOutputFolder;
    }

    private void printTick(long tick, long firstTimestamp, long lastTimeStamp, long startTime) {
        if (!this.verbose) {
            return;
        }
        double percentageComplete = (double)(tick - firstTimestamp) / (double)(lastTimeStamp - firstTimestamp) * 100.0;
        System.out.printf("Tick at %s (%d) - %.2f%% complete - %s elapsed\n", new Date(tick), tick, percentageComplete, TestDriver.getElaspsed(startTime));
    }

    private static String getElaspsed(long start) {
        double t = System.currentTimeMillis() - start;
        if (t < 1000.0) {
            return TestDriver.slf(t) + "ms";
        }
        if (t < 60000.0) {
            return TestDriver.slf(t / 1000.0) + "s " + TestDriver.slf(t % 1000.0) + "ms";
        }
        if (t < 3600000.0) {
            return TestDriver.slf(t / 60000.0) + "m " + TestDriver.slf(t % 60000.0 / 1000.0) + "s " + TestDriver.slf(t % 1000.0) + "ms";
        }
        if (t < 8.64E7) {
            return TestDriver.slf(t / 3600000.0) + "h " + TestDriver.slf(t % 3600000.0 / 60000.0) + "m " + TestDriver.slf(t % 60000.0 / 1000.0) + "s " + TestDriver.slf(t % 1000.0) + "ms";
        }
        return TestDriver.slf(t / 8.64E7) + "d " + TestDriver.slf(t % 8.64E7 / 3600000.0) + "h " + TestDriver.slf(t % 3600000.0 / 60000.0) + "m " + TestDriver.slf(t % 60000.0 / 1000.0) + "s " + TestDriver.slf(t % 1000.0) + "ms";
    }

    private static String slf(double n) {
        return String.valueOf(Double.valueOf(Math.floor(n)).longValue());
    }

    public static DriverBuilder builder() {
        return new DriverBuilder();
    }

    public static class DriverBuilder {
        private AlarmDatasource alarmDatasource;
        private InventoryDatasource inventoryDatasource;
        private SituationDatasource situationDatasource;
        private EngineFactory engineFactory;
        private Boolean verbose;
        private long graphExportIntervalMs = 0L;
        private File graphOutputFolder = new File("/tmp");

        public DriverBuilder withAlarmDatasource(AlarmDatasource alarmDatasource) {
            this.alarmDatasource = alarmDatasource;
            return this;
        }

        public DriverBuilder withInventoryDatasource(InventoryDatasource inventoryDatasource) {
            this.inventoryDatasource = inventoryDatasource;
            return this;
        }

        public DriverBuilder withSituationDatasource(SituationDatasource situationDatasource) {
            this.situationDatasource = situationDatasource;
            return this;
        }

        public DriverBuilder withEngineFactory(EngineFactory engineFactory) {
            this.engineFactory = engineFactory;
            return this;
        }

        public DriverBuilder withVerboseOutput() {
            this.verbose = true;
            return this;
        }

        public DriverBuilder withGraphExportIntervalMs(long graphExportIntervalMs) {
            this.graphExportIntervalMs = graphExportIntervalMs;
            return this;
        }

        public DriverBuilder withGraphOutputFolder(File graphOutputFolder) {
            this.graphOutputFolder = graphOutputFolder;
            return this;
        }

        public TestDriver build() {
            if (this.engineFactory == null) {
                throw new IllegalArgumentException("Engine factory is required.");
            }
            TestDriver driver = new TestDriver(this.engineFactory);
            if (this.verbose != null) {
                driver.setVerbose(this.verbose);
            }
            driver.setGraphExportIntervalMs(this.graphExportIntervalMs);
            driver.setGraphOutputFolder(this.graphOutputFolder);
            return driver;
        }
    }

    private class DriverSession
    implements SituationHandler {
        final Map<String, Situation> situations = new LinkedHashMap<String, Situation>();

        private DriverSession() {
        }

        public void onSituation(Situation situation) {
            if (TestDriver.this.verbose) {
                System.out.printf("Situation with id %s has %d alarms.\n", situation.getId(), situation.getAlarms().size());
            }
            this.situations.put(situation.getId() + ":" + situation.getSeverity(), situation);
        }
    }
}

