/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.alec.engine.cluster;

import edu.uci.ics.jung.graph.Graph;
import edu.uci.ics.jung.graph.SparseMultigraph;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.opennms.alec.datasource.api.Alarm;
import org.opennms.alec.datasource.api.InventoryObject;
import org.opennms.alec.datasource.api.InventoryObjectPeerRef;
import org.opennms.alec.datasource.api.InventoryObjectRelativeRef;
import org.opennms.alec.datasource.api.ResourceKey;
import org.opennms.alec.engine.cluster.CEEdge;
import org.opennms.alec.engine.cluster.CEVertex;
import org.opennms.alec.features.graph.api.Edge;
import org.opennms.alec.features.graph.api.GraphChangedListener;
import org.opennms.alec.features.graph.api.Vertex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GraphManager {
    private static final Logger LOG = LoggerFactory.getLogger(GraphManager.class);
    private final AtomicLong vertexIdGenerator = new AtomicLong();
    private final AtomicLong edgeIdGenerator = new AtomicLong();
    private final Map<Long, CEVertex> idtoVertexMap = new HashMap<Long, CEVertex>();
    private final Map<ResourceKey, CEVertex> resourceKeyVertexMap = new HashMap<ResourceKey, CEVertex>();
    private final Graph<CEVertex, CEEdge> g = new SparseMultigraph();
    private final AtomicBoolean didGraphChange = new AtomicBoolean();
    private final Set<Long> disconnectedVertices = new HashSet<Long>();
    private final Map<ResourceKey, Set<InventoryObject>> deferredIosByDependency = new HashMap<ResourceKey, Set<InventoryObject>>();
    private final Map<InventoryObject, Set<ResourceKey>> dependenciesByDeferredIos = new HashMap<InventoryObject, Set<ResourceKey>>();
    private final List<GraphChangedListener> graphChangedListeners = new CopyOnWriteArrayList<GraphChangedListener>();

    public synchronized void addInventory(Collection<InventoryObject> inventory) {
        this.addOrUpdateInventory(inventory);
    }

    private synchronized void addOrUpdateInventory(Collection<InventoryObject> inventory) {
        ResourceKey resourceKey;
        LinkedList<CEVertex> verticesAdded = new LinkedList<CEVertex>();
        LinkedList<CEVertex> verticesToVerify = new LinkedList<CEVertex>();
        AtomicBoolean haveWeChangedGraph = new AtomicBoolean(false);
        for (InventoryObject io : inventory) {
            resourceKey = GraphManager.getResourceKeyFor(io);
            CEVertex ioVertex = this.resourceKeyVertexMap.computeIfAbsent(resourceKey, key -> {
                LOG.trace("Adding vertex with resource key: {} for inventory object: {}", (Object)resourceKey, (Object)io);
                CEVertex vertex = this.createVertexFor(io);
                this.g.addVertex((Object)vertex);
                haveWeChangedGraph.set(true);
                this.idtoVertexMap.put(vertex.getNumericId(), vertex);
                verticesAdded.add(vertex);
                return vertex;
            });
            if (ioVertex.getInventoryObject().isEmpty()) {
                ioVertex.setInventoryObject(io);
            }
            verticesToVerify.add(ioVertex);
        }
        for (InventoryObject io : inventory) {
            CEEdge edge;
            resourceKey = GraphManager.getResourceKeyFor(io);
            CEVertex vertex = this.resourceKeyVertexMap.get(resourceKey);
            boolean didDeferRelation = false;
            ResourceKey parentResourceKey = GraphManager.getResourceKeyForParent(io);
            if (parentResourceKey != null) {
                CEVertex parentVertex = this.resourceKeyVertexMap.get(parentResourceKey);
                if (parentVertex == null) {
                    LOG.info("No existing vertex found for parent with resource key '{}' on vertex with resource key '{}'. Deferring edge association.", (Object)parentResourceKey, (Object)resourceKey);
                    this.defer(io, parentResourceKey);
                    didDeferRelation = true;
                } else if (!this.g.isNeighbor((Object)parentVertex, (Object)vertex)) {
                    LOG.trace("Adding edge between child: {} and parent: {}", (Object)vertex, (Object)parentVertex);
                    CEEdge edge2 = CEEdge.newParentEdge(this.edgeIdGenerator.getAndIncrement(), io.getWeightToParent());
                    this.g.addEdge((Object)edge2, (Object)parentVertex, (Object)vertex);
                    verticesToVerify.add(parentVertex);
                    haveWeChangedGraph.set(true);
                }
            }
            for (InventoryObjectPeerRef peerRef : io.getPeers()) {
                ResourceKey peerResourceKey = GraphManager.getResourceKeyForPeer(peerRef);
                CEVertex peerVertex = this.resourceKeyVertexMap.get(peerResourceKey);
                if (peerVertex == null) {
                    LOG.info("No existing vertex found for peer with resource key '{}' on vertex with resource key '{}'. Deferring edge association.", (Object)peerResourceKey, (Object)resourceKey);
                    this.defer(io, peerResourceKey);
                    didDeferRelation = true;
                    continue;
                }
                if (this.g.isNeighbor((Object)peerVertex, (Object)vertex)) continue;
                LOG.debug("Adding edge between peers A: {} and Z: {}", (Object)peerVertex, (Object)vertex);
                edge = CEEdge.newPeerEdge(this.edgeIdGenerator.getAndIncrement(), peerRef);
                this.g.addEdge((Object)edge, (Object)peerVertex, (Object)vertex);
                verticesToVerify.add(peerVertex);
                haveWeChangedGraph.set(true);
            }
            for (InventoryObjectRelativeRef relativeRef : io.getRelatives()) {
                ResourceKey relativeResourceKey = GraphManager.getResourceKeyForPeer(relativeRef);
                CEVertex relativeVertex = this.resourceKeyVertexMap.get(relativeResourceKey);
                if (relativeVertex == null) {
                    LOG.info("No existing vertex found for relative with resource key '{}' on vertex with resource key '{}'. Deferring edge association.", (Object)relativeResourceKey, (Object)resourceKey);
                    this.defer(io, relativeResourceKey);
                    didDeferRelation = true;
                    continue;
                }
                if (this.g.isNeighbor((Object)relativeVertex, (Object)vertex)) continue;
                LOG.debug("Adding edge between relatives A: {} and Z: {}", (Object)relativeVertex, (Object)vertex);
                edge = CEEdge.newRelativeEdge(this.edgeIdGenerator.getAndIncrement(), relativeRef);
                this.g.addEdge((Object)edge, (Object)relativeVertex, (Object)vertex);
                verticesToVerify.add(relativeVertex);
                haveWeChangedGraph.set(true);
            }
            if (didDeferRelation) continue;
            this.clearDeferralsFor(io);
        }
        this.handleDeferredIos(verticesAdded);
        this.trackDisconnectedVertices(verticesToVerify);
        if (haveWeChangedGraph.get()) {
            this.didGraphChange.set(true);
            this.notifyGraphChangedListeners();
        }
    }

    private void defer(InventoryObject io, ResourceKey ... waitingFor) {
        for (ResourceKey resourceKey : waitingFor) {
            this.deferredIosByDependency.compute(resourceKey, (k, v) -> {
                HashSet<InventoryObject> iosWaiting = v;
                if (iosWaiting == null) {
                    iosWaiting = new HashSet<InventoryObject>();
                }
                iosWaiting.add(io);
                return iosWaiting;
            });
        }
        this.dependenciesByDeferredIos.compute(io, (k, v) -> {
            HashSet<ResourceKey> dependencies = v;
            if (dependencies == null) {
                dependencies = new HashSet<ResourceKey>();
            }
            dependencies.addAll(Arrays.asList(waitingFor));
            return dependencies;
        });
    }

    private void handleDeferredIos(List<CEVertex> newVertices) {
        if (newVertices.isEmpty() || this.deferredIosByDependency.isEmpty()) {
            return;
        }
        Set<InventoryObject> iosToUpdate = newVertices.stream().flatMap(v -> this.deferredIosByDependency.getOrDefault(v.getResourceKey(), Collections.emptySet()).stream()).collect(Collectors.toSet());
        if (iosToUpdate.isEmpty()) {
            return;
        }
        this.addOrUpdateInventory(iosToUpdate);
    }

    private void clearDeferralsFor(InventoryObject io) {
        Set<ResourceKey> dependencies = this.dependenciesByDeferredIos.remove(io);
        if (dependencies == null || dependencies.isEmpty()) {
            return;
        }
        for (ResourceKey resourceKey : dependencies) {
            Set<InventoryObject> deferredIos = this.deferredIosByDependency.get(resourceKey);
            if (deferredIos == null) continue;
            deferredIos.remove(io);
            if (!deferredIos.isEmpty()) continue;
            this.deferredIosByDependency.remove(resourceKey);
        }
    }

    public synchronized void removeInventory(Collection<InventoryObject> inventory) {
        AtomicBoolean haveWeChangedGraph = new AtomicBoolean(false);
        for (InventoryObject io : inventory) {
            ResourceKey resourceKey = GraphManager.getResourceKeyFor(io);
            CEVertex vertex = this.resourceKeyVertexMap.remove(resourceKey);
            if (vertex != null) {
                Collection neighbors = this.g.getNeighbors((Object)vertex);
                this.g.removeVertex((Object)vertex);
                this.disconnectedVertices.remove(vertex.getNumericId());
                this.trackDisconnectedVertices(neighbors);
                haveWeChangedGraph.set(true);
            }
            this.clearDeferralsFor(io);
        }
        if (haveWeChangedGraph.get()) {
            this.didGraphChange.set(true);
            this.notifyGraphChangedListeners();
        }
    }

    private void trackDisconnectedVertices(Collection<CEVertex> verticesToVerify) {
        for (CEVertex v : verticesToVerify) {
            if (this.g.getNeighborCount((Object)v) == 0) {
                this.disconnectedVertices.add(v.getNumericId());
                continue;
            }
            this.disconnectedVertices.remove(v.getNumericId());
        }
    }

    public synchronized void addOrUpdateAlarms(List<Alarm> alarms) {
        for (Alarm alarm : alarms) {
            this.addOrUpdateAlarm(alarm);
        }
    }

    public synchronized Optional<CEVertex> addOrUpdateAlarm(Alarm alarm) {
        if (alarm.getInventoryObjectType() == null || alarm.getInventoryObjectId() == null) {
            LOG.info("Alarm with id: {} is not associated with any resource. It will not be added to the graph.", (Object)alarm.getId());
            return Optional.empty();
        }
        ResourceKey resourceKey = GraphManager.getResourceKeyFor(alarm);
        CEVertex vertex = this.resourceKeyVertexMap.computeIfAbsent(resourceKey, key -> {
            LOG.info("No existing vertex was found with resource key: {} for alarm with id: {} and contents: {}. Creating a new vertex.", new Object[]{resourceKey, alarm.getId(), alarm});
            CEVertex v = new CEVertex(this.vertexIdGenerator.getAndIncrement(), resourceKey);
            this.g.addVertex((Object)v);
            this.didGraphChange.set(true);
            this.idtoVertexMap.put(v.getNumericId(), v);
            this.handleDeferredIos(Collections.singletonList(v));
            return v;
        });
        LOG.trace("Updating vertex: {} with alarm: {}", (Object)vertex, (Object)alarm);
        vertex.addOrUpdateAlarm(alarm);
        this.notifyGraphChangedListeners();
        return Optional.of(vertex);
    }

    public <V> V withReadOnlyGraph(Function<Graph<? extends Vertex, ? extends Edge>, V> consumer) {
        return consumer.apply(this.g);
    }

    public void withReadOnlyGraph(Consumer<Graph<? extends Vertex, ? extends Edge>> consumer) {
        consumer.accept(this.g);
    }

    public synchronized <V> V withGraph(Function<Graph<CEVertex, CEEdge>, V> consumer) {
        return consumer.apply(this.g);
    }

    public synchronized void withGraph(Consumer<Graph<CEVertex, CEEdge>> consumer) {
        consumer.accept(this.g);
    }

    public synchronized void withVertex(String type, String id, BiConsumer<Graph<CEVertex, CEEdge>, CEVertex> consumer) {
        consumer.accept(this.g, this.resourceKeyVertexMap.get(ResourceKey.key((String[])new String[]{type, id})));
    }

    public int getVertexCount() {
        return this.g.getVertexCount();
    }

    protected Graph<CEVertex, CEEdge> getGraph() {
        return this.g;
    }

    protected CEVertex getVertexWithId(Long id) {
        return this.idtoVertexMap.get(id);
    }

    private CEVertex createVertexFor(InventoryObject io) {
        ResourceKey resourceKey = GraphManager.getResourceKeyFor(io);
        return new CEVertex(this.vertexIdGenerator.getAndIncrement(), resourceKey, io);
    }

    private static ResourceKey getResourceKeyFor(InventoryObject io) {
        return ResourceKey.key((String[])new String[]{io.getType(), io.getId()});
    }

    private static ResourceKey getResourceKeyForParentOf(InventoryObject io) {
        return ResourceKey.key((String[])new String[]{io.getParentType(), io.getParentId()});
    }

    private static ResourceKey getResourceKeyFor(Alarm alarm) {
        return ResourceKey.key((String[])new String[]{alarm.getInventoryObjectType(), alarm.getInventoryObjectId()});
    }

    private static ResourceKey getResourceKeyForParent(InventoryObject child) {
        if (child.getParentType() != null && child.getParentId() != null) {
            return ResourceKey.key((String[])new String[]{child.getParentType(), child.getParentId()});
        }
        return null;
    }

    private static ResourceKey getResourceKeyForPeer(InventoryObjectPeerRef peerRef) {
        return ResourceKey.key((String[])new String[]{peerRef.getType(), peerRef.getId()});
    }

    private static ResourceKey getResourceKeyForPeer(InventoryObjectRelativeRef relativeRef) {
        return ResourceKey.key((String[])new String[]{relativeRef.getType(), relativeRef.getId()});
    }

    public boolean getDidGraphChangeAndReset() {
        return this.didGraphChange.getAndSet(false);
    }

    public Set<Long> getDisconnectedVertices() {
        return this.disconnectedVertices;
    }

    public int getNumDeferredObjects() {
        return this.dependenciesByDeferredIos.size();
    }

    public Optional<CEVertex> getVertexFor(InventoryObject io) {
        ResourceKey resourceKey = GraphManager.getResourceKeyFor(io);
        return Optional.ofNullable(this.resourceKeyVertexMap.get(resourceKey));
    }

    public Optional<CEVertex> getVertexForParentOf(InventoryObject io) {
        ResourceKey resourceKey = GraphManager.getResourceKeyForParentOf(io);
        return Optional.ofNullable(this.resourceKeyVertexMap.get(resourceKey));
    }

    public void registerGraphChangedListener(GraphChangedListener listener) {
        if (!this.graphChangedListeners.contains(listener)) {
            this.graphChangedListeners.add(listener);
        }
    }

    private void notifyGraphChangedListeners() {
        for (GraphChangedListener listener : this.graphChangedListeners) {
            listener.graphHasChanged();
        }
    }
}

