/*******************************************************************************
 * This file is part of OpenNMS(R).
 *
 * Copyright (C) 2022 The OpenNMS Group, Inc.
 * OpenNMS(R) is Copyright (C) 1999-2022 The OpenNMS Group, Inc.
 *
 * OpenNMS(R) is a registered trademark of The OpenNMS Group, Inc.
 *
 * OpenNMS(R) is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * OpenNMS(R) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OpenNMS(R).  If not, see:
 *      http://www.gnu.org/licenses/
 *
 * For more information contact:
 *     OpenNMS(R) Licensing <license@opennms.org>
 *     http://www.opennms.org/
 *     http://www.opennms.com/
 *******************************************************************************/

package org.opennms.integration.api.xml.schema.eventconf;

import com.google.common.base.MoreObjects;
import org.opennms.integration.api.v1.config.events.AttributeType;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlType;
import java.io.File;
import java.io.Serializable;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

@XmlRootElement(name = "collectionGroup")
@XmlAccessorType(XmlAccessType.NONE)
@XmlType(propOrder = {"rrd", "collection"})
public class CollectionGroup implements Serializable {
    private static final long serialVersionUID = 2L;

    /**
     * Default folder where RRD files are stored.
     * <p>
     * Working directory defaults to $OPENNMS_HOME, so we can find these in $OPENNMS_HOME/share/rrd.
     */
    public static final File DEFAULT_RRD_ROOT = Paths.get("share", "rrd").toAbsolutePath().toFile();

    /**
     * Directory name of where all other collected data is stored.
     */
    public static final String SNMP_DIRECTORY = "snmp";

    @XmlAttribute(name = "name", required = true)
    private String name;

    @XmlAttribute(name = "resourceType")
    private String resourceType = "nodeSnmp";

    @XmlAttribute(name = "instance")
    private String instance;

    @XmlElement(name = "rrd")
    private Rrd rrd;

    @XmlElement(name = "collection")
    private List<Collection> collection = new ArrayList<>();

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getResourceType() {
        return resourceType;
    }

    public void setResourceType(String resourceType) {
        this.resourceType = resourceType;
    }

    public String getInstance() {
        return instance;
    }

    public void setInstance(String instance) {
        this.instance = instance;
    }

    public Rrd getRrd() {
        return this.rrd;
    }

    public void setRrd(final Rrd rrd) {
        this.rrd = rrd;
    }

    public List<Collection> getCollection() {
        return collection;
    }

    public void setCollection(List<Collection> collection) {
        this.collection = collection;
    }

    @Override
    public int hashCode() {
        return Objects.hash(name, resourceType, rrd);
    }

    @Override
    public boolean equals(final Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof CollectionGroup) {
            final CollectionGroup that = (CollectionGroup) obj;
            return Objects.equals(this.name, that.name) &&
                    Objects.equals(this.resourceType, that.resourceType) &&
                    Objects.equals(this.rrd, that.rrd) &&
                    Objects.equals(this.collection, that.collection);
        }
        return false;
    }

    @XmlRootElement(name = "rrd")
    @XmlAccessorType(XmlAccessType.NONE)
    public static class Rrd {
        private static final File DEFAULT_BASE_DIRECTORY = new File(DEFAULT_RRD_ROOT, SNMP_DIRECTORY);

        /**
         * Step size for the RRD, in seconds.
         */
        @XmlAttribute(name = "step")
        private Integer step;

        /**
         * HeartBeat of the RRD, default is step * 2
         */
        @XmlAttribute(name = "heartBeat")
        private Integer heartBeat = -1;

        /**
         * Round Robin Archive definitions
         */
        @XmlElement(name = "rra")
        private List<String> rras = new ArrayList<>();

        public Integer getStep() {
            return this.step;
        }

        public void setStep(final Integer step) {
            this.step = step;
        }

        public int getHeartBeat() {
            return heartBeat == -1 ? this.getStep() * 2 : heartBeat;
        }

        public void setHeartBeat(int heartBeat) {
            this.heartBeat = heartBeat;
        }

        public List<String> getRras() {
            return this.rras;
        }

        public void setRras(final List<String> rras) {
            this.rras = rras;
        }

        public File getBaseDir() {
            return DEFAULT_BASE_DIRECTORY;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            final Rrd that = (Rrd) o;
            return Objects.equals(this.step, that.step) &&
                    Objects.equals(this.rras, that.rras) &&
                    Objects.equals(this.heartBeat, that.heartBeat);
        }

        @Override
        public int hashCode() {
            return Objects.hash(this.step, this.rras, this.heartBeat);
        }

        @Override
        public String toString() {
            return MoreObjects.toStringHelper(this)
                    .add("step", this.step)
                    .add("rras", this.rras)
                    .add("heartBeat", this.heartBeat)
                    .toString();
        }
    }

    @XmlRootElement(name = "collection")
    @XmlAccessorType(XmlAccessType.NONE)
    public static class Collection {
        @XmlAttribute(name = "name", required = true)
        private String name;

        @XmlAttribute(name = "rename")
        private String rename;

        @XmlAttribute(name = "type")
        private AttributeType type;

        @XmlElement(name = "paramValue")
        private List<ParamValue> paramValue = new ArrayList<>();

        public String getName() {
            return name;
        }

        public void setName(String name) {
            this.name = name;
        }

        public String getRename() {
            return rename;
        }

        public void setRename(String rename) {
            this.rename = rename;
        }

        public AttributeType getType() {
            return type;
        }

        public void setType(AttributeType type) {
            this.type = type;
        }

        public List<ParamValue> getParamValue() {
            return paramValue;
        }

        public void setParamValue(List<ParamValue> paramValue) {
            this.paramValue = paramValue;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            final Collection that = (Collection) o;
            return Objects.equals(this.name, that.name) &&
                    Objects.equals(this.type, that.type) &&
                    Objects.equals(this.paramValue, that.paramValue);
        }

        @Override
        public int hashCode() {
            return Objects.hash(this.name, this.type, this.paramValue);
        }

        @Override
        public String toString() {
            return MoreObjects.toStringHelper(this)
                    .add("name", this.name)
                    .add("type", this.type)
                    .add("paramValue", this.paramValue)
                    .toString();
        }
    }

    @XmlRootElement(name = "paramValue")
    @XmlAccessorType(XmlAccessType.NONE)
    public static class ParamValue {
        @XmlAttribute(name = "key", required = true)
        private String name;

        @XmlAttribute(name = "value", required = true)
        private Double value;

        public String getName() {
            return name;
        }

        public void setName(String name) {
            this.name = name;
        }

        public Double getValue() {
            return value;
        }

        public void setValue(Double value) {
            this.value = value;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            final ParamValue that = (ParamValue) o;
            return Objects.equals(this.name, that.name) &&
                    Objects.equals(this.value, that.value);
        }

        @Override
        public int hashCode() {
            return Objects.hash(this.name, this.value);
        }

        @Override
        public String toString() {
            return MoreObjects.toStringHelper(this)
                    .add("name", this.name)
                    .add("type", this.value)
                    .toString();
        }
    }
}
