/*
 * Decompiled with CFR 0.152.
 */
package com.adobe.epubcheck.util;

import com.google.common.base.Function;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import java.nio.CharBuffer;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Pattern;
import org.w3c.epubcheck.util.infra.InfraUtil;

public final class SourceSet {
    private static final Function<ImageSource, String> LIST_TO_URL_FUNC = new Function<ImageSource, String>(){

        @Override
        public String apply(ImageSource input) {
            return input.url;
        }
    };
    private final List<ImageSource> sources;
    private final List<String> urls;
    private final String size;

    public static SourceSet parse(String srcset) {
        return SourceSet.parse(srcset, null);
    }

    public static SourceSet parse(String srcset, ErrorHandler errorHandler) {
        return new Parser(errorHandler).parse(srcset);
    }

    private SourceSet(List<ImageSource> sources, String size) {
        this.sources = Preconditions.checkNotNull(sources);
        this.size = size;
        this.urls = Lists.transform(sources, LIST_TO_URL_FUNC);
    }

    public List<ImageSource> getImageSources() {
        return this.sources;
    }

    public List<String> getImageURLs() {
        return this.urls;
    }

    public String getSourceSize() {
        return this.size;
    }

    public boolean isEmpty() {
        return this.sources.isEmpty();
    }

    public static interface ErrorHandler {
        public void error(ParseError var1, int var2);
    }

    private static final class Parser {
        private static Pattern POSITIVE_FLOAT = Pattern.compile("(\\d+(\\.\\d+)?|\\.\\d+)([eE][-+]?\\d+)?");
        private final ErrorHandler errorHandler;

        public Parser(ErrorHandler errorHandler) {
            this.errorHandler = errorHandler;
        }

        public SourceSet parse(CharSequence srcset) {
            Builder builder = new Builder();
            if (srcset == null || srcset.length() == 0) {
                this.error(ParseError.NULL_OR_EMPTY, -1);
                return builder.build();
            }
            CharBuffer input = CharBuffer.wrap(srcset);
            LinkedList<String> descriptors = new LinkedList<String>();
            StringBuilder url = new StringBuilder();
            StringBuilder descriptor = new StringBuilder();
            State state = State.SPLIT;
            boolean consume = true;
            char c = ' ';
            while (!consume || input.hasRemaining()) {
                if (consume) {
                    c = input.get();
                } else {
                    consume = true;
                }
                switch (state) {
                    case SPLIT: {
                        assert (url.length() == 0);
                        if (InfraUtil.isASCIIWhitespace(c)) break;
                        if (c == ',') {
                            if (!builder.hasSources()) {
                                this.error(ParseError.EMPTY_START, input.position());
                                break;
                            }
                            this.error(ParseError.EMPTY_MIDDLE, input.position());
                            break;
                        }
                        state = State.COLLECT_URL;
                        consume = false;
                        break;
                    }
                    case COLLECT_URL: {
                        if (InfraUtil.isASCIIWhitespace(c)) {
                            if (url.charAt(url.length() - 1) == ',') {
                                builder.add(this.finalizeSource(url, descriptors));
                                state = State.SPLIT;
                                break;
                            }
                            state = State.TOKENIZE_DESCRIPTORS;
                            break;
                        }
                        url.append(c);
                        break;
                    }
                    case TOKENIZE_DESCRIPTORS: {
                        if (InfraUtil.isASCIIWhitespace(c)) break;
                        state = State.IN_DESCRIPTOR;
                        consume = false;
                        break;
                    }
                    case IN_DESCRIPTOR: {
                        if (InfraUtil.isASCIIWhitespace(c)) {
                            if (descriptor.length() > 0) {
                                descriptors.add(descriptor.toString());
                                descriptor.setLength(0);
                            }
                            state = State.AFTER_DESCRIPTOR;
                            break;
                        }
                        if (c == ',') {
                            if (descriptor.length() > 0) {
                                descriptors.add(descriptor.toString());
                                descriptor.setLength(0);
                            }
                            builder.add(this.finalizeSource(url, descriptors));
                            state = State.SPLIT;
                            break;
                        }
                        if (c == '(') {
                            descriptor.append(c);
                            state = State.IN_PARENS;
                            break;
                        }
                        descriptor.append(c);
                        break;
                    }
                    case IN_PARENS: {
                        if (c == ')') {
                            descriptor.append(c);
                            state = State.IN_DESCRIPTOR;
                            break;
                        }
                        descriptor.append(c);
                        break;
                    }
                    case AFTER_DESCRIPTOR: {
                        if (InfraUtil.isASCIIWhitespace(c)) break;
                        state = State.IN_DESCRIPTOR;
                        consume = false;
                    }
                }
            }
            if (descriptor.length() > 0) {
                descriptors.add(descriptor.toString());
            }
            builder.add(this.finalizeSource(url, descriptors));
            return builder.build();
        }

        private static int removeTrailingCommas(StringBuilder sb) {
            int count = 0;
            while (sb.charAt(sb.length() - 1) == ',') {
                sb.deleteCharAt(sb.length() - 1);
                ++count;
            }
            return count;
        }

        private ImageSource finalizeSource(StringBuilder url, List<String> descriptors) {
            if (url.length() == 0) {
                return null;
            }
            int trailingCommas = Parser.removeTrailingCommas(url);
            if (trailingCommas > 1) {
                this.error(ParseError.EMPTY_MIDDLE, -1);
            }
            ParseError error = null;
            int width = -1;
            float density = -1.0f;
            int futureH = -1;
            for (String descriptor : descriptors) {
                char last = descriptor.charAt(descriptor.length() - 1);
                char first = descriptor.charAt(0);
                String value = descriptor.substring(0, descriptor.length() - 1);
                if (first == '-' || first == '+') {
                    error = ParseError.DESCRIPTOR_WIDTH_SIGNED;
                    continue;
                }
                if (last == 'w') {
                    if (width != -1 || density != -1.0f) {
                        error = width != -1 ? ParseError.DESCRIPTOR_WIDTH_MORE_THAN_ONCE : ParseError.DESCRIPTOR_MIX_WIDTH_DENSITY;
                        continue;
                    }
                    try {
                        width = Integer.parseInt(value);
                        if (width != 0) continue;
                        error = ParseError.DESCRIPTOR_WIDTH_ZERO;
                        width = -1;
                    }
                    catch (NumberFormatException ex) {
                        error = ParseError.DESCRIPTOR_WIDTH_NOT_INT;
                    }
                    continue;
                }
                if (last == 'x') {
                    if (width != -1 || density != -1.0f || futureH != -1) {
                        error = density != -1.0f ? ParseError.DESCRIPTOR_DENSITY_MORE_THAN_ONCE : ParseError.DESCRIPTOR_MIX_WIDTH_DENSITY;
                        continue;
                    }
                    try {
                        if (!POSITIVE_FLOAT.matcher(value).matches()) {
                            throw new NumberFormatException();
                        }
                        density = Float.parseFloat(value);
                        if (!(density < 0.0f)) continue;
                        error = ParseError.DESCRIPTOR_DENSITY_NEGATIVE;
                        density = -1.0f;
                    }
                    catch (NumberFormatException ex) {
                        error = ParseError.DESCRIPTOR_DENSITY_NOT_FLOAT;
                    }
                    continue;
                }
                if (last == 'h') {
                    if (futureH != -1 || density != -1.0f) {
                        error = futureH != -1 ? ParseError.DESCRIPTOR_FUTURE_H_MORE_THAN_ONCE : ParseError.DESCRIPTOR_FUTURE_H_WITH_DENSITY;
                        continue;
                    }
                    try {
                        futureH = Integer.parseInt(value);
                        if (futureH != 0) continue;
                        error = ParseError.DESCRIPTOR_FUTURE_H_ZERO;
                        futureH = -1;
                    }
                    catch (NumberFormatException ex) {
                        error = ParseError.DESCRIPTOR_FUTURE_H_NOT_INT;
                    }
                    continue;
                }
                error = ParseError.DESCRIPTOR_UNKNOWN_SUFFIX;
            }
            if (futureH != -1 && width == -1) {
                error = ParseError.DESCRIPTOR_FUTURE_H_WITHOUT_WIDTH;
            }
            String source = url.toString();
            url.setLength(0);
            descriptors.clear();
            if (error != null) {
                this.error(error, -1);
                return null;
            }
            return ImageSource.create(source, width, density);
        }

        private void error(ParseError code, int position) {
            if (this.errorHandler != null) {
                this.errorHandler.error(code, position);
            }
        }

        private static enum State {
            SPLIT,
            COLLECT_URL,
            TOKENIZE_DESCRIPTORS,
            IN_DESCRIPTOR,
            AFTER_DESCRIPTOR,
            IN_PARENS;

        }
    }

    private static final class Builder {
        private final List<ImageSource> sources = new LinkedList<ImageSource>();
        private String size = null;

        private Builder() {
        }

        public Builder add(ImageSource source) {
            if (source != null) {
                this.sources.add(source);
            }
            return this;
        }

        public Builder size(String size) {
            this.size = size;
            return this;
        }

        public boolean hasSources() {
            return !this.sources.isEmpty();
        }

        public SourceSet build() {
            return new SourceSet(ImmutableList.copyOf(this.sources), Strings.nullToEmpty(this.size));
        }
    }

    public static enum ParseError {
        NULL_OR_EMPTY,
        EMPTY_START,
        EMPTY_MIDDLE,
        DESCRIPTOR_MIX_WIDTH_DENSITY,
        DESCRIPTOR_DENSITY_MORE_THAN_ONCE,
        DESCRIPTOR_DENSITY_NEGATIVE,
        DESCRIPTOR_DENSITY_NOT_FLOAT,
        DESCRIPTOR_WIDTH_MORE_THAN_ONCE,
        DESCRIPTOR_WIDTH_SIGNED,
        DESCRIPTOR_WIDTH_NOT_INT,
        DESCRIPTOR_WIDTH_ZERO,
        DESCRIPTOR_FUTURE_H_ZERO,
        DESCRIPTOR_FUTURE_H_MORE_THAN_ONCE,
        DESCRIPTOR_FUTURE_H_NOT_INT,
        DESCRIPTOR_FUTURE_H_WITHOUT_WIDTH,
        DESCRIPTOR_FUTURE_H_WITH_DENSITY,
        DESCRIPTOR_UNKNOWN_SUFFIX;

    }

    public static final class ImageSource {
        public final String url;
        public final Optional<Integer> width;
        public final Optional<Float> density;

        private ImageSource(String url, Optional<Integer> width, Optional<Float> density) {
            this.url = url;
            this.width = width;
            this.density = density;
        }

        public static ImageSource create(String url, int width, float density) {
            Preconditions.checkArgument(!Strings.isNullOrEmpty(url));
            return new ImageSource(url, Optional.fromNullable(width < 0 ? null : Integer.valueOf(width)), Optional.fromNullable(density < 0.0f ? null : Float.valueOf(density)));
        }
    }
}

