/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { RuntimeError } from '../errors';
import { setInjectorProfilerContext } from '../render3/debug/injector_profiler';
import { getCurrentInjector, setCurrentInjector } from './injector_compatibility';
import { getInjectImplementation, setInjectImplementation } from './inject_switch';
import { R3Injector } from './r3_injector';
/**
 * Runs the given function in the [context](guide/dependency-injection-context) of the given
 * `Injector`.
 *
 * Within the function's stack frame, [`inject`](api/core/inject) can be used to inject dependencies
 * from the given `Injector`. Note that `inject` is only usable synchronously, and cannot be used in
 * any asynchronous callbacks or after any `await` points.
 *
 * @param injector the injector which will satisfy calls to [`inject`](api/core/inject) while `fn`
 *     is executing
 * @param fn the closure to be run in the context of `injector`
 * @returns the return value of the function, if any
 * @publicApi
 */
export function runInInjectionContext(injector, fn) {
    if (injector instanceof R3Injector) {
        injector.assertNotDestroyed();
    }
    let prevInjectorProfilerContext;
    if (ngDevMode) {
        prevInjectorProfilerContext = setInjectorProfilerContext({ injector, token: null });
    }
    const prevInjector = setCurrentInjector(injector);
    const previousInjectImplementation = setInjectImplementation(undefined);
    try {
        return fn();
    }
    finally {
        setCurrentInjector(prevInjector);
        ngDevMode && setInjectorProfilerContext(prevInjectorProfilerContext);
        setInjectImplementation(previousInjectImplementation);
    }
}
/**
 * Asserts that the current stack frame is within an [injection
 * context](guide/dependency-injection-context) and has access to `inject`.
 *
 * @param debugFn a reference to the function making the assertion (used for the error message).
 *
 * @publicApi
 */
export function assertInInjectionContext(debugFn) {
    // Taking a `Function` instead of a string name here prevents the unminified name of the function
    // from being retained in the bundle regardless of minification.
    if (!getInjectImplementation() && !getCurrentInjector()) {
        throw new RuntimeError(-203 /* RuntimeErrorCode.MISSING_INJECTION_CONTEXT */, ngDevMode &&
            (debugFn.name +
                '() can only be used within an injection context such as a constructor, a factory function, a field initializer, or a function used with `runInInjectionContext`'));
    }
}
//# sourceMappingURL=data:application/json;base64,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