/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { createMultiResultQuerySignalFn, createSingleResultOptionalQuerySignalFn, createSingleResultRequiredQuerySignalFn } from '../render3/query_reactive';
function viewChildFn(locator, opts) {
    return createSingleResultOptionalQuerySignalFn();
}
function viewChildRequiredFn(locator, opts) {
    return createSingleResultRequiredQuerySignalFn();
}
/**
 * Initializes a view child query.
 *
 * Consider using `viewChild.required` for queries that should always match.
 *
 * @usageNotes
 * Create a child query in your component by declaring a
 * class field and initializing it with the `viewChild()` function.
 *
 * ```ts
 * @Component({template: '<div #el></div><my-component #cmp />'})
 * export class TestComponent {
 *   divEl = viewChild<ElementRef>('el');                   // Signal<ElementRef|undefined>
 *   divElRequired = viewChild.required<ElementRef>('el');  // Signal<ElementRef>
 *   cmp = viewChild(MyComponent);                          // Signal<MyComponent|undefined>
 *   cmpRequired = viewChild.required(MyComponent);         // Signal<MyComponent>
 * }
 * ```
 *
 * @developerPreview
 */
export const viewChild = (() => {
    // Note: This may be considered a side-effect, but nothing will depend on
    // this assignment, unless this `viewChild` constant export is accessed. It's a
    // self-contained side effect that is local to the user facing `viewChild` export.
    viewChildFn.required = viewChildRequiredFn;
    return viewChildFn;
})();
/**
 * Initializes a view children query.
 *
 * Query results are represented as a signal of a read-only collection containing all matched
 * elements.
 *
 * @usageNotes
 * Create a children query in your component by declaring a
 * class field and initializing it with the `viewChildren()` function.
 *
 * ```ts
 * @Component({...})
 * export class TestComponent {
 *   divEls = viewChildren<ElementRef>('el');   // Signal<ReadonlyArray<ElementRef>>
 * }
 * ```
 */
export function viewChildren(locator, opts) {
    return createMultiResultQuerySignalFn();
}
export function contentChildFn(locator, opts) {
    return createSingleResultOptionalQuerySignalFn();
}
function contentChildRequiredFn(locator, opts) {
    return createSingleResultRequiredQuerySignalFn();
}
/**
 * Initializes a content child query. Consider using `contentChild.required` for queries that should
 * always match.
 *
 * @usageNotes
 * Create a child query in your component by declaring a
 * class field and initializing it with the `contentChild()` function.
 *
 * ```ts
 * @Component({...})
 * export class TestComponent {
 *   headerEl = contentChild<ElementRef>('h');                    // Signal<ElementRef|undefined>
 *   headerElElRequired = contentChild.required<ElementRef>('h'); // Signal<ElementRef>
 *   header = contentChild(MyHeader);                             // Signal<MyHeader|undefined>
 *   headerRequired = contentChild.required(MyHeader);            // Signal<MyHeader>
 * }
 * ```
 */
export const contentChild = (() => {
    // Note: This may be considered a side-effect, but nothing will depend on
    // this assignment, unless this `viewChild` constant export is accessed. It's a
    // self-contained side effect that is local to the user facing `viewChild` export.
    contentChildFn.required = contentChildRequiredFn;
    return contentChildFn;
})();
/**
 * Initializes a content children query.
 *
 * Query results are represented as a signal of a read-only collection containing all matched
 * elements.
 *
 * @usageNotes
 * Create a children query in your component by declaring a
 * class field and initializing it with the `contentChildren()` function.
 *
 * ```ts
 * @Component({...})
 * export class TestComponent {
 *   headerEl = contentChildren<ElementRef>('h');   // Signal<ReadonlyArray<ElementRef>>
 * }
 * ```
 */
export function contentChildren(locator, opts) {
    return createMultiResultQuerySignalFn();
}
//# sourceMappingURL=data:application/json;base64,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