/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { assertInInjectionContext, DestroyRef, effect, inject, Injector, untracked } from '@angular/core';
import { ReplaySubject } from 'rxjs';
/**
 * Exposes the value of an Angular `Signal` as an RxJS `Observable`.
 *
 * The signal's value will be propagated into the `Observable`'s subscribers using an `effect`.
 *
 * `toObservable` must be called in an injection context unless an injector is provided via options.
 *
 * @developerPreview
 */
export function toObservable(source, options) {
    !options?.injector && assertInInjectionContext(toObservable);
    const injector = options?.injector ?? inject(Injector);
    const subject = new ReplaySubject(1);
    const watcher = effect(() => {
        let value;
        try {
            value = source();
        }
        catch (err) {
            untracked(() => subject.error(err));
            return;
        }
        untracked(() => subject.next(value));
    }, { injector, manualCleanup: true });
    injector.get(DestroyRef).onDestroy(() => {
        watcher.destroy();
        subject.complete();
    });
    return subject.asObservable();
}
//# sourceMappingURL=data:application/json;base64,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