// For Library Version: 1.120.26

declare module "sap/ui/export/library" {
  /**
   * EDM data types for document export.
   *
   * @since 1.50.0
   */
  export enum EdmType {
    /**
     * Property of type BigNumber.
     *
     * @since 1.60
     */
    BigNumber = "BigNumber",
    /**
     * Property of type Boolean.
     */
    Boolean = "Boolean",
    /**
     * Property of type Currency
     */
    Currency = "Currency",
    /**
     * Property of type Date.
     */
    Date = "Date",
    /**
     * Property of type DateTime.
     */
    DateTime = "DateTime",
    /**
     * Property of type Enumeration.
     *
     * @since 1.58
     */
    Enumeration = "Enumeration",
    /**
     * Property of type Number.
     */
    Number = "Number",
    /**
     * Property of type Percentage.
     *
     * @since 1.87
     */
    Percentage = "Percentage",
    /**
     * Property of type string.
     */
    String = "String",
    /**
     * Property of type Time.
     */
    Time = "Time",
    /**
     * Property of type Timezone
     *
     * @since 1.118
     */
    Timezone = "Timezone",
  }
  /**
   * File types for document export.
   *
   * @since 1.78
   */
  export enum FileType {
    /**
     * Portable Document Format (PDF) file type.
     *
     * @since 1.117
     */
    PDF = "PDF",
    /**
     * Office Open XML - SpreadsheetML file type.
     */
    XLSX = "XLSX",
  }
}

declare module "sap/ui/export/ExportBase" {
  import EventProvider from "sap/ui/base/EventProvider";

  import ListBinding from "sap/ui/model/ListBinding";

  import { FileType } from "sap/ui/export/library";

  import Metadata from "sap/ui/base/Metadata";

  import TreeBinding from "sap/ui/model/TreeBinding";

  import Event from "sap/ui/base/Event";

  /**
   * The `sap.ui.export.ExportBase` class allows you to export table data from a UI5 application to certain
   * formats. This class is an abstract class that requires specific implementations for each file format.
   *
   * @since 1.96
   */
  export default class ExportBase extends EventProvider {
    /**
     * Base class for specific SAPUI5 export implementations. This class contains abstract functions that need
     * to be implemented.
     */
    constructor(
      /**
       * Export settings
       */
      mSettings: {
        /**
         * Data and formatting related export settings
         */
        workbook: object;
        /**
         * Source of export data. A data source properties map or `sap.ui.model.ListBinding` can be provided. An
         * instance of `sap.ui.model.ListBinding` has to implement a `#getDownloadUrl` function.
         */
        dataSource: object | ListBinding;
        /**
         * The maximal number of records to export
         */
        count?: int;
        /**
         * Optional file name for the exported file
         */
        fileName?: string;
        /**
         * `FileType` that is used to identify the file-ending and MIME-type of the file
         */
        fileType?: FileType | keyof typeof FileType;
      }
    );

    /**
     * Creates a new subclass of class sap.ui.export.ExportBase with name `sClassName` and enriches it with
     * the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.base.EventProvider.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ExportBase>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.ui.export.ExportBase.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
    /**
     * Attaches event handler `fnFunction` to the {@link sap.ui.export.ExportBase#event:beforeExport} event
     * of this `sap.ui.export.ExportBase`.
     *  When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.export.ExportBase` itself.
     *  This event is fired just before the export process is started.
     *
     * @since 1.96
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachBeforeExport(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnHandler: Function,
      /**
       * Context object to call the event handler with. Defaults to the `sap.ui.export.ExportBase` instance itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link sap.ui.export.ExportBase#event:beforeExport} event
     * of this `sap.ui.export.ExportBase`.
     *  When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.export.ExportBase` itself.
     *  This event is fired just before the export process is started.
     *
     * @since 1.96
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachBeforeExport(
      /**
       * The function to be called when the event occurs
       */
      fnHandler: Function,
      /**
       * Context object to call the event handler with. Defaults to the `sap.ui.export.ExportBase` instance itself
       */
      oListener?: object
    ): this;
    /**
     * Triggers the export process of the specific format.
     *
     *
     * @returns Promise that gets resolved once the data has been exported
     */
    build(): Promise<any>;
    /**
     * Cancels the current export process.
     */
    cancel(): void;
    /**
     * Cleans up the internal structures and removes all event handlers.
     *
     * The object must not be used anymore after destroy was called.
     * See:
     * 	sap.ui.base.Object#destroy
     */
    destroy(): void;
    /**
     * Detaches event handler `fnFunction` from the {@link sap.ui.export.ExportBase#event:beforeExport} event
     * of this `sap.ui.export.ExportBase`.
     *  The passed function and listener object must match the ones used for event registration.
     *
     * @since 1.96
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachBeforeExport(
      /**
       * The function to be called when the event occurs
       */
      fnHandler: Function,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Returns the specific MIME type
     *
     * @since 1.112
     */
    getMimeType(): void;
    /**
     * Sets the data source configuration that will be used for exporting the data. If the passed parameter
     * is null, the call will be ignored.
     *
     *
     * @returns - Valid dataSource object or null in case the dataSource configuration is not supported
     */
    processDataSource(
      /**
       * Possible types are a data source configuration, a `sap.ui.model.ListBinding` or `sap.ui.model.TreeBinding`
       */
      oDataSource: object | ListBinding | TreeBinding
    ): object | null;
  }
  /**
   * Parameters of the ExportBase#beforeExport event.
   */
  export interface ExportBase$BeforeExportEventParameters {}

  /**
   * Event object of the ExportBase#beforeExport event.
   */
  export type ExportBase$BeforeExportEvent = Event<
    ExportBase$BeforeExportEventParameters,
    ExportBase
  >;
}

declare module "sap/ui/export/ExportHandler" {
  import EventProvider from "sap/ui/base/EventProvider";

  import Metadata from "sap/ui/base/Metadata";

  import Filter from "sap/ui/export/util/Filter";

  import Event from "sap/ui/base/Event";

  /**
   * The `sap.ui.export.ExportHandler` class allows you to export table data from an SAPUI5 application.
   *
   * @since 1.102
   */
  export default class ExportHandler extends EventProvider {
    /**
     * Any export-related functionality is encapsulated in the `ExportHandler` that also stores user settings
     * throughout the session.
     */
    constructor(
      /**
       * `ExportHandler` capabilities that define supported file types and features
       */
      mCapabilities: object
    );

    /**
     * Creates a new subclass of class sap.ui.export.ExportHandler with name `sClassName` and enriches it with
     * the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.base.EventProvider.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ExportHandler>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.ui.export.ExportHandler.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
    /**
     * Attaches event handler `fnFunction` to the {@link sap.ui.export.ExportHandler#event:beforeExport} event
     * of this `sap.ui.export.ExportHandler`.
     *  When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.export.ExportHandler` itself.
     *  This event is fired just before the export process is started.
     *
     * @since 1.102
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachBeforeExport(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnHandler: Function,
      /**
       * Context object to call the event handler with; defaults to this `sap.ui.export.ExportHandler` instance
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link sap.ui.export.ExportHandler#event:beforeExport} event
     * of this `sap.ui.export.ExportHandler`.
     *  When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.export.ExportHandler` itself.
     *  This event is fired just before the export process is started.
     *
     * @since 1.102
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachBeforeExport(
      /**
       * The function to be called when the event occurs
       */
      fnHandler: Function,
      /**
       * Context object to call the event handler with; defaults to this `sap.ui.export.ExportHandler` instance
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Cleans up the internal structures and removes all event handlers.
     *
     * The object must not be used anymore after destroy was called.
     * See:
     * 	sap.ui.base.Object#destroy
     */
    destroy(): void;
    /**
     * Detaches event handler `fnFunction` from the {@link sap.ui.export.ExportHandler#event:beforeExport} event
     * of this `sap.ui.export.ExportHandler`.
     *  The passed function and listener object must match the ones used for event registration.
     *
     * @since 1.102
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachBeforeExport(
      /**
       * The function to be called when the event occurs
       */
      fnHandler: Function,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Exports the data as defined via parameter. The function returns a `Promise` that will be resolved after
     * the export process has been finished. In case of an error, its message will be shown in a `Dialog`, and
     * the `Promise` will be rejected.
     *
     * @since 1.102
     *
     * @returns A `Promise` that gets resolved after the export process has been finished
     */
    export(
      /**
       * Export settings that are used for the export
       */
      mExportSettings: object
    ): Promise<any>;
    /**
     * Exports the data as defined by the user. This function will show an export settings dialog where the
     * user can define certain settings that influence the output of the export functionality.
     *
     * @since 1.102
     *
     * @returns A `Promise` that resolves once the data has been exported
     */
    exportAs(
      /**
       * General export settings containing `workbook` and `dataSource` information
       */
      oSettings: object,
      /**
       * Resolves the label for a particular column that is not directly contained in the export settings
       */
      fnResolveColumnLabel?: Function
    ): Promise<any>;
  }
  /**
   * Parameters of the ExportHandler#beforeExport event.
   */
  export interface ExportHandler$BeforeExportEventParameters {
    /**
     * Contains export-related configuration
     */
    exportSettings?: object;

    /**
     * User-specific settings from the Export As dialog
     */
    userExportSettings?: object;

    /**
     * Array of filter settings for the exported data
     */
    filterSettings?: Filter[];
  }

  /**
   * Event object of the ExportHandler#beforeExport event.
   */
  export type ExportHandler$BeforeExportEvent = Event<
    ExportHandler$BeforeExportEventParameters,
    ExportHandler
  >;
}

declare module "sap/ui/export/Spreadsheet" {
  import ExportBase from "sap/ui/export/ExportBase";

  import ListBinding from "sap/ui/model/ListBinding";

  import TreeBinding from "sap/ui/model/TreeBinding";

  import Event from "sap/ui/base/Event";

  /**
   * The `sap.ui.export.Spreadsheet` class allows you to export table data from a UI5 application to a spreadsheet
   * file.
   *
   * @since 1.50
   */
  export default class Spreadsheet extends ExportBase {
    /**
     * Creates a new spreadsheet export object. Use this object to build and download a spreadsheet file in
     * Office Open XML Spreadsheet format from tabular data. This functionality is normally used together with
     * UI5 tables.
     *
     * Overview: The class builds a spreadsheet in an Office Open XML Spreadsheet format using tabular data
     * from a specified data source. Data is retrieved and the document is built asynchronously in a worker
     * thread of the browser. The status of the process is visually presented to the user in a progress dialog
     * that can be suppressed. The user can cancel the process with the Cancel button of the dialog.
     *
     * This class provides a low level API for spreadsheet export. The {@link sap.ui.comp.smarttable.SmartTable }
     * control implements it internally and provides the export functionality out of the box. For special cases,
     * please refer to details below.
     *
     * Optional features:
     * 	 - Suppress the progress dialog.
     * 	 - Suppress worker and run the document generation process in a main thread.
     * 	 - Configure the exported file name.
     *
     * Export settings object: Export settings should be provided in the constructor as an `mSettings` property
     * map with the following fields:
     * 	 - `workbook` - Spreadsheet properties object
     * 	`workbook.columns` - Array of column configurations. Each column configuration is an object with the
     * following fields:
     * 	`label` (string) - Column header text
     * 	 - `property` (string) - Field name or Array of field names in the data source feed
     * 	 - `type` (string) - Optional data type of the field. See {@link sap.ui.export.EdmType} for the list
     *     of supported types. If this property is omitted, the property is processed as a string field.
     * 	 - `width` (number) - Optional width of the column in characters. There is no 1:1 correspondence between
     *     character widths in the exported spreadsheet and CSS units.The width of one character is approximately
     *     0.5em in CSS units, depending on the fonts that are used in the table and in the resulting spreadsheet.
     *     The default value is 10 characters.
     * 	 - `textAlign` (string) - Horizontal alignment of cell contents. The following values of the CSS `text-align`
     *     property are accepted: `[left, right, center, begin, end]`. If not specified, the columns are horizontally
     *     aligned based on the property type.
     * 	 - `scale` (number) - Number of digits after decimal point for numeric values
     * 	 - `delimiter` (boolean) - Set to `true` to display thousands separators in numeric values. The default
     *     value is `false`.
     * 	 - `unit` (string) - Text to display as the unit of measurement or currency next to the numeric value.
     *     It is treated as a string and has no influence on the value itself. For example, a value of 150 with
     *     the unit "%" is still 150 and not 1.5, as a user may expect.
     * 	 - `unitProperty` (string) - Name of the data source field that contains the unit/currency text
     * 	 - `displayUnit` (boolean) - The property applies to currency values only and defines if the currency
     *     is shown in the column. The default value is `true`.
     * 	 - `trueValue` (string) - Textual representation of a boolean type that has the value `true`
     * 	 - `falseValue` (string) - Textual representation of a boolean type that has the value `false`
     * 	 - `template` (string) - Formatting template that supports indexed placeholders within curly brackets
     *
     * 	 - `inputFormat` (string) - Formatting template for string formatted dates
     * 	 - `utc` (boolean) - Defines whether the `DateTime` is displayed as UTC or local time
     * 	 - `valueMap` (string) - Mapping object or Map containing the values that should be mapped to a particular
     *     key
     * 	 - `wrap` (boolean) - Indicates if wrapping is enabled for this particular column
     * 	 - `workbook.context` - Context object that will be applied to the generated file. It may contain the
     *     following fields:
     * 	 - `application` (string) - The application that creates the XLSX document (default: "SAP UI5")
     * 	 - `version` (string) - Application version that creates the XLSX document (default: "${version}")
     * 	 - `title` (string) - Title of the XLSX document (NOT the filename)
     * 	 - `modifiedBy` (string) - User context for the XLSX document
     * 	 - `sheetName` (string) - The label of the data sheet
     * 	 - `metaSheetName` (string) - The label of the metadata sheet. The sheet will not be shown unless metadata
     *     entries are provided
     * 	 - `metainfo` (Array) - An Array of metadata groups. Each group has a name property and an items Array
     *     which contains key/value pairs
     * 	 - `workbook.hierarchyLevel` - Name of the property that contains the hierarchy level information of
     *     each line item
     * 	 - `dataSource` - Source of spreadsheet data. It can be a JSON array with row data, an URL or an OData
     *     properties object with the following fields:
     * 	`type` (string) - Type of the data source. Currently, only OData is supported and the value have to
     * be set to `"odata"`.
     * 	 - `dataUrl` (string) - URL to table data on the server, including all select, filter, and search query
     *     parameters
     * 	 - `serviceUrl` (string) - URL to the OData service. The parameter is required for OData batch requests.
     *
     * 	 - `count` (number) - Count of available records on the server
     * 	 - `useBatch` (boolean) - Set to `true` if OData batch requests are used to fetch the spreadsheet data.
     *     In this case, `serviceUrl` and `headers` have to be specified, too.
     * 	 - `headers` (object) - Map of HTTP request header properties. They should correspond to the HTTP request
     *     headers that are used to obtain table data for display in the browser.
     * 	 - `sizeLimit` (number) - Maximal allowed number of records that can be obtained from the service in
     *     a single request
     * 	 - `count` (number) - The maximal number of records to export. If not specified, all data from the data
     *     source is fetched.
     * 	 - `worker` (boolean) - Run export process in a worker thread. Set to `false` to disable worker and
     *     run export in a main thread. This is needed, for example, if a mock server is used to provide spreadsheet
     *     data.
     *  **Note:** In case of a strict content security policy, it is not always possible to create an export
     * worker. In this case, export runs in a main thread disregarding the `worker` value.
     * 	 - `fileName` (string) - Optional file name for the exported file. If not specified, the spreadsheet
     *     is exported as `export.xlsx`.
     * 	 - `showProgress` (boolean) - Set to `false` to suppress the progress dialog
     *
     * Usage: To start export, create a new `sap.ui.export.Spreadsheet` object and call the `build` method.
     * Column configuration, data source, and export settings must be provided in the constructor. The `build`
     * method opens a progress dialog and starts an asynchronous export process. The export process fetches
     * data rows from the data source, builds a spreadsheet in-browser in a worker thread, and finally downloads
     * the document to the client.
     *
     * Example:
     * ```javascript
     *
     *   var oSpreadsheet = new sap.ui.export.Spreadsheet(mSettings);
     *   oSpreadsheet.build();
     * ```
     *
     *
     * Optionally, you can attach `onprogress` event listeners to be notified about the export progress and
     * follow the completion status of the returned `Promise`.
     *
     * Example:
     * ```javascript
     *
     *   var oSpreadsheet = new sap.ui.export.Spreadsheet(mSettings);
     *   oSpreadsheet.onprogress = function(iValue) {
     *   	{@link sap.base.Log#debug Log.debug}("Export: %" + iValue + " completed");
     *   };
     *   oSpreadsheet.build()
     *     .then( function() { {@link sap.base.Log#debug Log.debug}("Export is finished"); })
     *     .catch( function(sMessage) { {@link sap.base.Log#error Log.error}("Export error: " + sMessage); });
     * ```
     *
     *
     * Example of column configuration:
     * ```javascript
     *
     *   var aColumns = [];
     *   aColumns.push({
     *   	label: "Name",
     *   	property: "name"
     *   });
     *   aColumns.push({
     *     label: "Salary",
     *     property: "salary",
     *     type: "number",
     *     scale: 2
     *   });
     *
     *   var mSettings = {
     *     workbook: {
     *       columns: aColumns,
     *       context: {
     *         application: 'Debug Test Application',
     *         version: '${version}',
     *         title: 'Some random title',
     *         modifiedBy: 'John Doe',
     *         metaSheetName: 'Custom metadata',
     *         metainfo: [
     *           {
     *             name: 'Grouped Properties',
     *             items: [
     *               { key: 'administrator', value: 'Foo Bar' },
     *               { key: 'user', value: 'John Doe' },
     *               { key: 'server', value: 'server.domain.local' }
     *             ]
     *           },
     *           {
     *             name: 'Another Group',
     *             items: [
     *               { key: 'property', value: 'value' },
     *               { key: 'some', value: 'text' },
     *               { key: 'fu', value: 'bar' }
     *             ]
     *           }
     *         ]
     *       },
     *       hierarchyLevel: 'level'
     *     },
     *     dataSource: mDataSource,
     *     fileName: "salary.xlsx"
     *   };
     *   var oSpreadsheet = new sap.ui.export.Spreadsheet(mSettings);
     *   oSpreadsheet.build();
     * ```
     *
     *
     * Restrictions: For a complete list of restrictions, see: {@link https://ui5.sap.com/#/topic/2c641481649f44de9c1c22c9c3c49d13 Spreadsheet Export Restrictions}
     *
     * You can export only the primitive cell data types that are listed in {@link sap.ui.export.EdmType}. Icons,
     * images, check boxes, and complex controls in UI5 table cells are not supported.
     *
     * Custom formatters in data binding are not supported.
     *
     * The size of an exported table is limited by available browser memory. Export of large data sets can lead
     * to memory overflow errors. Therefore, do not use `sap.ui.export.Spreadsheet` with data tables containing
     * more than 2,000,000 table cells on desktop computers and more than 100,000 cells on mobile devices. Consider
     * a specialized export solution in such cases. For example, MS Excel® can import spreadsheets from an OData
     * services directly, without any UI.
     *
     * The export process runs in a worker thread whenever possible. However, code injection to native XMLHttpRequest
     * events is not available in the worker environment. Therefore, the `worker` parameter in export settings
     * should be set to `false` if the application uses a mock server to fetch table data.
     *
     * For exporting hierarchy level information, the maximum hierarchy depth is 8. This restriction results
     * from the Office Open XML standard and the programs that can open such files. The sap.ui.export.Spreadsheet
     * allows you to export more hierarchy levels although they might not be displayed correctly when opening
     * the generated file if the hierarchy depth exceeds the value of 8.
     *
     * The column configuration must contain at least one column to execute the export process. If there is
     * no column configured, the export will be canceled.
     *
     * If the export is used within a table, any row that is showing aggregated data (i.E. sum row) will not
     * be exported.
     *
     * The properties sheetName and metaSheetName on the workbook.context object are limited to 31 characters
     * each. If their value exceeds this maximum length, the value will be truncated.
     * See:
     * 	{@link https://ui5.sap.com/#/topic/2691788a08fc43f7bf269ea7c6336caf Spreadsheet}
     */
    constructor(
      /**
       * Export settings
       */
      mSettings: {
        /**
         * Spreadsheet properties
         */
        workbook: {
          /**
           * Column configuration
           */
          columns: any[];
          /**
           * Export context that will be applied to the exported file
           */
          context?: {
            /**
             * Application that created this XLSX
             */
            application?: string;
            /**
             * Application version that was used to create this XLSX
             */
            version?: string;
            /**
             * Title of the XLSX document (NOT the file name)
             */
            title?: string;
            /**
             * User context for the exported document
             */
            modifiedBy?: string;
            /**
             * The name of the data sheet that will be shown in Excel
             */
            sheetName?: string;
            /**
             * The name of the metadata sheet that will be shown in Excel
             */
            metaSheetName?: string;
            /**
             * Optional Metadata that will be displayed in the additional Metadata Sheet
             */
            metainfo?: any[];
          };
          /**
           * Optional name of the property that contains hierarchy level information
           */
          hierarchyLevel?: string;
        };
        /**
         * Source of spreadsheet data. A JSON array, data source properties map, `sap.ui.model.ListBinding`, `sap.ui.model.TreeBinding`
         * or URL to an OData source can be provided. For example, `"someUrl"` is an equivalent to `{dataUrl:"someUrl",
         * type:"OData"}`. An instance of `sap.ui.model.ListBinding` or `sap.ui.model.TreeBinding` either has to
         * implement a `#getDownloadUrl` function or needs to be a ClientListBinding. **Note:** `sap.ui.model.ClientTreeBinding`
         * is not supported.
         */
        dataSource:
          | string
          | {
              /**
               * Maximal allowed number of records that can be obtained from the service in a single request
               */
              sizeLimit: int;
            }
          | any[]
          | ListBinding
          | TreeBinding;
        /**
         * The maximal number of records to export
         */
        count?: int;
        /**
         * Run export process in a worker thread. Set to `false` to disable worker and run export in a main thread.
         * This is needed, for example, if a mock server is used to provide spreadsheet data.
         *  **Note:** In case of a strict content security policy, it is not always possible to create an export
         * worker. In this case, export runs in a main thread disregarding the `worker` value.
         */
        worker?: boolean;
        /**
         * Optional file name for the exported file
         */
        fileName?: string;
        /**
         * Set to `false` to suppress the progress dialog
         */
        showProgress?: boolean;
      }
    );

    /**
     * Attaches event handler `fnFunction` to the {@link sap.ui.export.Spreadsheet#event:beforeSave} event of
     * this `sap.ui.export.Spreadsheet`.
     *  When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.export.Spreadsheet` itself.
     *  This event is fired just before the generated file is saved to the file system.
     *
     * @since 1.61
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachBeforeSave(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnHandler: Function,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.export.Spreadsheet` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link sap.ui.export.Spreadsheet#event:beforeSave} event of
     * this `sap.ui.export.Spreadsheet`.
     *  When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.export.Spreadsheet` itself.
     *  This event is fired just before the generated file is saved to the file system.
     *
     * @since 1.61
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachBeforeSave(
      /**
       * The function to be called when the event occurs
       */
      fnHandler: Function,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.export.Spreadsheet` itself
       */
      oListener?: object
    ): this;
    /**
     * Cancels a running export process. This method does nothing if no export is running.
     *
     * @since 1.52
     *
     * @returns - Reference to `this` in order to allow method chaining
     */
    cancel(): this;
    /**
     * Detaches event handler `fnFunction` from the {@link sap.ui.export.Spreadsheet beforeSave} event of this
     * `sap.ui.export.Spreadsheet`.
     *  The passed function and listener object must match the ones used for event registration.
     *
     * @since 1.61
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachBeforeSave(
      /**
       * The function to be called, when the event occurs
       */
      fnHandler: Function,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Returns the specific MIME type
     *
     * @since 1.112
     *
     * @returns MIME type for Office Open XML Spreadsheet
     */
    getMimeType(): string;
    /**
     * Sets the data source configuration that will be used for exporting the data. If the passed parameter
     * is null, the call will be ignored.
     *
     * @since 1.73
     *
     * @returns - Valid dataSource object or null in case the dataSource configuration is not supported
     */
    processDataSource(
      /**
       * Possible types are a plain string that contains an URL of an OData service, an array of JSON objects,
       * a data source configuration, a `sap.ui.model.ListBinding` or `sap.ui.model.TreeBinding`
       */
      oDataSource: string | any[] | Object | ListBinding | TreeBinding
    ): object | null;
  }
  /**
   * Parameters of the Spreadsheet#beforeSave event.
   */
  export interface Spreadsheet$BeforeSaveEventParameters {}

  /**
   * Event object of the Spreadsheet#beforeSave event.
   */
  export type Spreadsheet$BeforeSaveEvent = Event<
    Spreadsheet$BeforeSaveEventParameters,
    Spreadsheet
  >;
}

declare module "sap/ui/export/util/Filter" {
  import BaseObject from "sap/ui/base/Object";

  import Metadata from "sap/ui/base/Metadata";

  import SimpleType from "sap/ui/model/SimpleType";

  /**
   * The `sap.ui.export.util.Filter` class represents filter settings that are used for the export. It provides
   * the capability to have a visual representation of the filters in the exported document and offers convenience
   * functions like `sap.ui.export.util.Filter#setType` to improve the result.
   *
   * @since 1.110
   */
  export default class Filter extends BaseObject {
    /**
     * Representation of filter settings that are used for exporting.
     */
    constructor(
      /**
       * Name of the technical filter property
       */
      sProperty: string,
      /**
       * Raw filter object
       */
      vRawFilter:
        | {
            /**
             * Filter operator
             */
            operator: string;
            /**
             * Filter value(s)
             */
            value: string | string[];
            /**
             * Defines whether it is an exclude filter
             */
            exclude?: boolean;
          }
        | Array<{
            /**
             * Filter operator
             */
            operator: string;
            /**
             * Filter value(s)
             */
            value: string | string[];
            /**
             * Defines whether it is an exclude filter
             */
            exclude?: boolean;
          }>,
      /**
       * Optional label for the filter property
       */
      sLabel?: string
    );

    /**
     * Creates a new subclass of class sap.ui.export.util.Filter with name `sClassName` and enriches it with
     * the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.base.Object.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, Filter>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.ui.export.util.Filter.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
    /**
     * Returns the filter label if available. Otherwise the name of the filter property will be returned.
     *
     * @since 1.110
     *
     * @returns Filter label
     */
    getLabel(): string;
    /**
     * Returns the technical name of the property on which the filter is applied.
     *
     * @since 1.110
     *
     * @returns Name of the property
     */
    getProperty(): string;
    /**
     * Returns the formatted filter value(s) as string. If there are multiple filters for the same property,
     * which are combined via `OR`, it will return a semicolon-separated list of the filter values including
     * their operators.
     *
     * @since 1.110
     *
     * @returns Formatted and semicolon-separated filter settings
     */
    getValue(): string;
    /**
     * Uses the given format function to format all filter values. The function has to accept a single parameter
     * of type `string` and needs to return a string value. The function will be called for every single raw
     * value without the corresponding filter operator.
     *
     * If there is a "between" filter, the function will be called twice. In case of an error, the function
     * call will be skipped, and the raw value will be taken instead.
     *
     * The format function has priority over the type-dependent format. To reset the format function and return
     * to type-dependent formatting, the format function needs to be set to `null`.
     *
     * @since 1.110
     */
    setFormat(
      /**
       * Format function that will be applied to the raw values
       */
      fnFormat: Function
    ): void;
    /**
     * Takes the given string as label of the filter.
     *
     * @since 1.110
     */
    setLabel(
      /**
       * Filter label
       */
      sLabel: string
    ): void;
    /**
     * Uses the given `sap.ui.model.SimpleType` instance to format the filter values accordingly.
     *
     * @since 1.110
     */
    setType(
      /**
       * Type instance that is used for formatting
       */
      oType: SimpleType
    ): void;
  }
}

declare namespace sap {
  interface IUI5DefineDependencyNames {
    "sap/ui/export/ExportBase": undefined;

    "sap/ui/export/ExportHandler": undefined;

    "sap/ui/export/ExportUtils": undefined;

    "sap/ui/export/library": undefined;

    "sap/ui/export/Spreadsheet": undefined;

    "sap/ui/export/util/Filter": undefined;
  }
}
