// For Library Version: 1.120.26

declare module "sap/ndc/library" {}

declare module "sap/ndc/BarcodeScanner" {
  import JSONModel from "sap/ui/model/json/JSONModel";

  /**
   * Please refer to SAP
   * Note 2402585 for information on Barcode Scanner support in native iOS and Android browsers.
   *
   * Here is an example of how to trigger the scan function of BarcodeScanner:
   * ```javascript
   *
   * sap.ui.require(["sap/ndc/BarcodeScanner"], function(BarcodeScanner) {
   * 	BarcodeScanner.scan(
   * 		function (oResult) { / * process scan result * / },
   * 		function (oError) { / * handle scan error * / },
   * 		function (oResult) { / * handle input dialog change * / }
   * 	);
   * });
   * ```
   *  The Barcode Scanner control can open the flashlight of Android devices if supported. The Barcode Scanner
   * control can also scan the barcode from a selected image file in gallery, or a photo taking by System
   * Camera App directly.
   *
   * The Barcode Scanner control integrates with the laser scanner when the page is loaded in the Enterprise
   * Browser on a Zebra device. To enable laser scanning with a Zebra device, two JavaScript files (ebapi.js
   * and eb.barcode.js) need to be loaded during runtime by the Enterprise Browser.
   * 	 - Your company admin / IT should configure the Barcode API settings in the Enterprise Browser config.xml
   *     file using mobile device management (MDM). Refer to CustomDOMElements
   *     for detailed information (recommended).
   * 	 - Developers can load these files directly into an HTML file. Refer to Enabling the API for detailed
   *     information.
   * 	 - Developers can refer to the KBA
   *     3390217 for guidance on configuring Zebra Enterprise Browser.
   *
   * @since 1.28.0
   */
  interface BarcodeScanner {
    /**
     * Closes the barcode input dialog. It can be used to close the dialog before the user presses the OK or
     * the Cancel button (e.g. in the fnLiveUpdate callback function of the {@link sap.ndc.BarcodeScanner.scan }
     * method.)
     */
    closeScanDialog(): void;
    /**
     * Disable the Bluetooth Barcode Scanner.
     */
    disableBluetoothBarcodeScanner(): void;
    /**
     * Set the callback function and settings to the Bluetooth Barcode Scanner.
     *
     * IMPORTANT: If the BarcodeScanner button is included into one iframe, the focus point must be into the
     * window of the iframe. If not, the Bluetooth Barcode Scanner cannot be triggered.
     *
     *
     * @returns Return True if the callback function or the settings is set successfully.
     */
    enableBluetoothBarcodeScanner(
      /**
       * Function to be called by the Bluetooth Barcode Scanner. The status of Bluetooth Barcode Scanner is enable
       * only when the callback function is set successfully.
       */
      bluetoothCallBack: Function,
      /**
       * An optional settings object to be used for the Bluetooth Barcode Scanner.
       *
       * Example:
       * ```javascript
       *
       *
       * {
       *     "handleFocusedElement": true, // The default value is false. If true, the scanning output will only be returned by the callback function and won't be entered into the input field or text area that is the focused element.
       *     "triggerOneTime": true, // The default value is false. If set to true, the Bluetooth Barcode Scanner is used only once for the next scanning.
       *     "scanningMode": "TimeInterval", // Either "TimeInterval" or "PrefixSuffix" scanning modes are available. The default value is "TimeInterval". If "TimeInterval" is selected, the scanner will search for barcodes where the maximum time interval between any two characters cannot be greater than the value of "intervalForDetection". If "PrefixSuffix" is selected, the scanner will search for barcodes that start with the prefix and end with the suffix based on the values of "prefix" and "suffix."
       *     "intervalForDetection": 80, // The default value is 80 milliseconds. In "TimeInterval" scanning mode, it represents the maximum time interval between two characters in the scanning data.
       *     "prefix": "$", // The default value is "$". In "PrefixSuffix" scanning mode, the parameter is utilized to find a match with the prefix of the received barcode data.
       *     "suffix": "#", // The default value is "#". In "PrefixSuffix" scanning mode, the parameter is utilized to find a match with the suffix of the received barcode data.
       *     "replaceGS1Separator": true, // The default value is false. Enabling this option in our function allows for the detection and replacement of special characters within GS1 barcodes using the value specified in the GS1ReplacementCharacter parameter. To support GS1 barcode, the scanner needs to be configured to support special character using model given by GS1FunctionKey parameter.
       *     "GS1FunctionKey": "altKey" // Either "altKey" or "ctrlKey" GS1 Function Key are available. The default value is "altKey". If "altKey" is selected, when the scanner is configured to support "ALt 3 Digit HEX" and the "[GS]" is present in the scanning data, the value "ALT + 029" will be obtained and then converted to the string "/x1D". If "ctrlKey" is selected, when the scanner is configured to support "Windows Mode Control + X" and the "[GS]" is present in the scanning data, the value "Ctrl + ]" will be obtained and then converted to the string "/x1D".
       *     "GS1ReplacementCharacter": undefined // The default value is undefined. By default, the [GS] will be replaced with the symbol string "/x1D". If the GS1ReplacementCharacter parameter value is available, the "[GS]" will be replaced by the value.
       * }
       *
       * ```
       */
      oSettings?: object
    ): boolean;
    /**
     * Returns the scanner API info that will be used to scan the barcode.
     *
     *
     * @returns The Barcode Scanner API info. (e.g. ZebraEnterpriseBrowser, Cordova, ZXingCPP or unknown)
     */
    getScanAPIInfo(): string;
    /**
     * Returns the status model of the Barcode Scanner. It is a JSON model which contains below properties:
     *
     * ```javascript
     *
     * {
     * 		scannerAPI: "ZXingCPP",
     * 		available: true,
     * 		deviceId: undefined,
     * 		devices: [],
     * 		apis: [
     * 			{
     * 				key: "ZebraEnterpriseBrowser",
     * 				status: "UnAvailable"
     * 			},
     * 			{
     * 				key: "Cordova",
     * 				status: "UnAvailable"
     * 			},
     * 			{
     * 				key: "ZXingCPP",
     * 				status: "Available"
     * 			}
     * 		]
     * 	}
     * ```
     *  '`scannerAPI`' shows the current scanner API used to scan the Barcode. '`available`' indicating whether
     * or not the Barcode Scanner feature is available. It can be used to bind to the `visible` property of
     * UI controls which have to be hidden in case the feature is unavailable. '`deviceId`' lists the current
     * used camera id of current device. Not working for iOS devices since do not support to get all the cameras.
     * '`devices`' lists all the cameras of current device. Not working for iOS devices since do not support
     * to get all the cameras. '`apis`' lists scanner APIs with status value. Status value can be: "Initial",
     * "Loading", "Available" or "UnAvailable".
     *
     * IMPORTANT: This model just shows current status of Barcode Scanner. Any change to it will not impact
     * Barcode Scanner.
     *
     *
     * @returns The Barcode Scanner Status Model
     */
    getStatusModel(): JSONModel;
    /**
     * Starts the barcode scanning process either showing the live input from the camera or displaying a dialog
     * to enter the value directly if the barcode scanning feature is unavailable.
     *
     * The barcode scanning is done asynchronously. When it is triggered, this function returns without waiting
     * for the scanning process to finish. The applications have to provide callback functions to react to the
     * events of a successful scanning, an error during scanning, and the live input on the dialog.
     *
     * `fnSuccess` is passed an object with text, format and cancelled properties. Text is the text representation
     * of the barcode data, format is the type of the barcode detected, and cancelled is whether or not the
     * user cancelled the scan. `fnError` is given the error, `fnLiveUpdate` is passed the new value entered
     * in the dialog's input field. An example:
     *
     *
     * ```javascript
     *
     * sap.ui.require(["sap/ndc/BarcodeScanner"], function(BarcodeScanner) {
     * 	BarcodeScanner.scan(
     * 		function (mResult) {
     * 			alert("We got a barcode\n" +
     * 			 	"Result: " + mResult.text + "\n" +
     * 			 	"Format: " + mResult.format + "\n" +
     * 			 	"Cancelled: " + mResult.cancelled);
     * 		},
     * 		function (Error) {
     * 			alert("Scanning failed: " + Error);
     * 		},
     * 		function (mParams) {
     * 			alert("Value entered: " + mParams.newValue);
     * 		},
     * 		"Enter Product Barcode",
     * 		true,
     * 		30,
     * 		1,
     * 		false,
     * 		false
     * 	);
     * });
     * ```
     */
    scan(
      /**
       * Function to be called when the scanning is done or cancelled
       */
      fnSuccess?: Function,
      /**
       * Function to be called when the scanning is failed
       */
      fnFail?: Function,
      /**
       * Function to be called when value of the dialog's input is changed
       */
      fnLiveUpdate?: Function,
      /**
       * Defines the barcode input dialog title. If unset, a predefined title will be used.
       */
      dialogTitle?: string,
      /**
       * Flag, which defines whether the front or back camera should be used.
       */
      preferFrontCamera?: boolean,
      /**
       * Defines the frame rate of the camera.
       */
      frameRate?: float,
      /**
       * Defines the zoom of the camera. This parameter is not supported on iOS.
       */
      zoom?: float,
      /**
       * Flag, which defines whether the camera should be used for scanning in Zebra Enterprise Browser.
       */
      keepCameraScan?: boolean,
      /**
       * Flag, which defines whether the Barcode input dialog should be shown.
       */
      disableBarcodeInputDialog?: boolean
    ): void;
    /**
     * Set the configs of the control Barcode Scanner.
     */
    setConfig(
      /**
       * The options are the configs that will be used to scan.
       *
       * Example:
       * ```javascript
       *
       *
       * {
       *     "enableGS1Header": true,  //If set to true, add the symbology identifier (GS1 specification 5.4.3.7. and 5.4.6.4.) as prefix into the result text.
       *     "deviceId": "string" // The specific camera id to scan the Barcode. If set to "", Barcode Scanner will use default camera. This option is not working for iOS devices since do not support to get all the cameras.
       *     "multiScan": {
       *         "enabled": false, // If set to true, will support scan multi barcodes, and the max number of the barcodes in each scanning is 10. Default value is false. Only working for ZXingCPP.
       *         "showPauseButton": false, // If set to true, will show a button so that can pause/restart the scan in the scan dialog.
       *         "pauseIfHasResult": false, // If set to true, will pause current scan dialog if has result. Only working if showPauseButton === false.
       *         "stopIfOnlyHasOneResult": false // If set to true, will stop current scan and return the result when only has one result. Only working if showPauseButton === false and pauseIfHasResult === true.
       *     }
       * }
       *
       * ```
       */
      options?: object
    ): void;
    /**
     * Set the sId of the element to be focused, which will then be used to retrieve the instance of the focused
     * SAP UI5 element.
     *
     *
     * @returns Return True if set success.
     */
    setFocusedInstanceId(
      /**
       * Defines the sId for the focused element.
       */
      sId?: string
    ): boolean;
    /**
     * Set the callback function for the physical scan button.
     */
    setPhysicalScan(
      /**
       * Function to be called when the scanning is done by pressing physical scan button.
       */
      fnPhysicalScan?: Function
    ): void;
    /**
     * Set the scanner API info that will be used to scan the barcode.
     *
     * IMPORTANT: The status of the scanner API must be **"Available"**(for ZXingCPP, status is **NOT "UnAvailable"**),
     * or will return False. Scanner APIs with status value can be got by using {@link #getStatusModel}. By
     * default, Barcode Scanner will select the scanner API(Available) with priority: ZebraEnterpriseBrowser
     * > Cordova > ZXingCPP.
     *
     *
     * @returns Return True if set success.
     */
    setScanAPIInfo(
      /**
       * Defines the scanner API to scan the barcode. Scanner API can be "ZebraEnterpriseBrowser", "Cordova" or
       * "ZXingCPP".
       */
      scannerAPI?: string
    ): boolean;
  }
  const BarcodeScanner: BarcodeScanner;
  export default BarcodeScanner;
}

declare module "sap/ndc/BarcodeScannerButton" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import Event from "sap/ui/base/Event";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { CSSSize } from "sap/ui/core/library";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * A button control (displaying a barcode icon) to start the barcode scanning process. If the native scanning
   * feature is not available or camera capability is not granted, the button is either hidden or it provides
   * a fallback by opening a dialog with an input field where the barcode can be entered manually.
   */
  export default class BarcodeScannerButton extends Control {
    /**
     * Constructor for a new BarcodeScannerButton.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $BarcodeScannerButtonSettings
    );
    /**
     * Constructor for a new BarcodeScannerButton.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $BarcodeScannerButtonSettings
    );

    /**
     * Creates a new subclass of class sap.ndc.BarcodeScannerButton with name `sClassName` and enriches it with
     * the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, BarcodeScannerButton>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.ndc.BarcodeScannerButton.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:inputLiveUpdate inputLiveUpdate} event of this
     * `sap.ndc.BarcodeScannerButton`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ndc.BarcodeScannerButton` itself.
     *
     * Event is fired when the text in the dialog's input field is changed.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachInputLiveUpdate(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: BarcodeScannerButton$InputLiveUpdateEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ndc.BarcodeScannerButton` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:inputLiveUpdate inputLiveUpdate} event of this
     * `sap.ndc.BarcodeScannerButton`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ndc.BarcodeScannerButton` itself.
     *
     * Event is fired when the text in the dialog's input field is changed.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachInputLiveUpdate(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: BarcodeScannerButton$InputLiveUpdateEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ndc.BarcodeScannerButton` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:scanFail scanFail} event of this `sap.ndc.BarcodeScannerButton`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ndc.BarcodeScannerButton` itself.
     *
     * Event is fired when the native scanning process is failed.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachScanFail(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ndc.BarcodeScannerButton` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:scanFail scanFail} event of this `sap.ndc.BarcodeScannerButton`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ndc.BarcodeScannerButton` itself.
     *
     * Event is fired when the native scanning process is failed.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachScanFail(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ndc.BarcodeScannerButton` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:scanSuccess scanSuccess} event of this `sap.ndc.BarcodeScannerButton`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ndc.BarcodeScannerButton` itself.
     *
     * Event is fired when the scanning is finished or cancelled
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachScanSuccess(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: BarcodeScannerButton$ScanSuccessEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ndc.BarcodeScannerButton` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:scanSuccess scanSuccess} event of this `sap.ndc.BarcodeScannerButton`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ndc.BarcodeScannerButton` itself.
     *
     * Event is fired when the scanning is finished or cancelled
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachScanSuccess(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: BarcodeScannerButton$ScanSuccessEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ndc.BarcodeScannerButton` itself
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:inputLiveUpdate inputLiveUpdate} event of
     * this `sap.ndc.BarcodeScannerButton`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachInputLiveUpdate(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: BarcodeScannerButton$InputLiveUpdateEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:scanFail scanFail} event of this `sap.ndc.BarcodeScannerButton`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachScanFail(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:scanSuccess scanSuccess} event of this `sap.ndc.BarcodeScannerButton`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachScanSuccess(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: BarcodeScannerButton$ScanSuccessEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:inputLiveUpdate inputLiveUpdate} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireInputLiveUpdate(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: BarcodeScannerButton$InputLiveUpdateEventParameters
    ): this;
    /**
     * Fires event {@link #event:scanFail scanFail} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireScanFail(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:scanSuccess scanSuccess} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireScanSuccess(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: BarcodeScannerButton$ScanSuccessEventParameters
    ): this;
    /**
     * Gets current value of property {@link #getDialogTitle dialogTitle}.
     *
     * Defines the barcode input dialog title. If unset, a predefined title will be used.
     *
     *
     * @returns Value of property `dialogTitle`
     */
    getDialogTitle(): string;
    /**
     * Gets current value of property {@link #getDisableBarcodeInputDialog disableBarcodeInputDialog}.
     *
     * If the value is true, the Barcode input dialog should be hidden.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `disableBarcodeInputDialog`
     */
    getDisableBarcodeInputDialog(): boolean;
    /**
     * Gets current value of property {@link #getFrameRate frameRate}.
     *
     * Defines the frame rate of the camera.
     *
     *
     * @returns Value of property `frameRate`
     */
    getFrameRate(): float;
    /**
     * Gets current value of property {@link #getKeepCameraScan keepCameraScan}.
     *
     * If set to true, the camera should be used for scanning in Zebra Enterprise Browser.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `keepCameraScan`
     */
    getKeepCameraScan(): boolean;
    /**
     * Gets current value of property {@link #getPreferFrontCamera preferFrontCamera}.
     *
     * If set to true, the front camera will be used to decode.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `preferFrontCamera`
     */
    getPreferFrontCamera(): boolean;
    /**
     * Gets current value of property {@link #getProvideFallback provideFallback}.
     *
     * If set to true, the button remains visible if the scanner is not available and triggers a dialog to enter
     * barcode.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `provideFallback`
     */
    getProvideFallback(): boolean;
    /**
     * Gets current value of property {@link #getWidth width}.
     *
     * Defines the width of the scanner button.
     *
     *
     * @returns Value of property `width`
     */
    getWidth(): CSSSize;
    /**
     * Gets current value of property {@link #getZoom zoom}.
     *
     * Defines the zoom of the camera. This parameter is not supported on iOS.
     *
     *
     * @returns Value of property `zoom`
     */
    getZoom(): float;
    /**
     * Sets a new value for property {@link #getDialogTitle dialogTitle}.
     *
     * Defines the barcode input dialog title. If unset, a predefined title will be used.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setDialogTitle(
      /**
       * New value for property `dialogTitle`
       */
      sDialogTitle?: string
    ): this;
    /**
     * Sets a new value for property {@link #getDisableBarcodeInputDialog disableBarcodeInputDialog}.
     *
     * If the value is true, the Barcode input dialog should be hidden.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setDisableBarcodeInputDialog(
      /**
       * New value for property `disableBarcodeInputDialog`
       */
      bDisableBarcodeInputDialog?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getFrameRate frameRate}.
     *
     * Defines the frame rate of the camera.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setFrameRate(
      /**
       * New value for property `frameRate`
       */
      fFrameRate: float
    ): this;
    /**
     * Sets a new value for property {@link #getKeepCameraScan keepCameraScan}.
     *
     * If set to true, the camera should be used for scanning in Zebra Enterprise Browser.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setKeepCameraScan(
      /**
       * New value for property `keepCameraScan`
       */
      bKeepCameraScan?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getPreferFrontCamera preferFrontCamera}.
     *
     * If set to true, the front camera will be used to decode.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setPreferFrontCamera(
      /**
       * New value for property `preferFrontCamera`
       */
      bPreferFrontCamera?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getProvideFallback provideFallback}.
     *
     * If set to true, the button remains visible if the scanner is not available and triggers a dialog to enter
     * barcode.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setProvideFallback(
      /**
       * New value for property `provideFallback`
       */
      bProvideFallback?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getWidth width}.
     *
     * Defines the width of the scanner button.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setWidth(
      /**
       * New value for property `width`
       */
      sWidth?: CSSSize
    ): this;
    /**
     * Sets a new value for property {@link #getZoom zoom}.
     *
     * Defines the zoom of the camera. This parameter is not supported on iOS.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setZoom(
      /**
       * New value for property `zoom`
       */
      fZoom: float
    ): this;
  }
  /**
   * Describes the settings that can be provided to the BarcodeScannerButton constructor.
   */
  export interface $BarcodeScannerButtonSettings extends $ControlSettings {
    /**
     * If set to true, the button remains visible if the scanner is not available and triggers a dialog to enter
     * barcode.
     */
    provideFallback?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines the width of the scanner button.
     */
    width?: CSSSize | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines the barcode input dialog title. If unset, a predefined title will be used.
     */
    dialogTitle?: string | PropertyBindingInfo;

    /**
     * If set to true, the front camera will be used to decode.
     */
    preferFrontCamera?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines the frame rate of the camera.
     */
    frameRate?: float | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines the zoom of the camera. This parameter is not supported on iOS.
     */
    zoom?: float | PropertyBindingInfo | `{${string}}`;

    /**
     * If set to true, the camera should be used for scanning in Zebra Enterprise Browser.
     */
    keepCameraScan?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * If the value is true, the Barcode input dialog should be hidden.
     */
    disableBarcodeInputDialog?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Event is fired when the scanning is finished or cancelled
     */
    scanSuccess?: (oEvent: BarcodeScannerButton$ScanSuccessEvent) => void;

    /**
     * Event is fired when the native scanning process is failed.
     */
    scanFail?: (oEvent: Event) => void;

    /**
     * Event is fired when the text in the dialog's input field is changed.
     */
    inputLiveUpdate?: (
      oEvent: BarcodeScannerButton$InputLiveUpdateEvent
    ) => void;
  }

  /**
   * Parameters of the BarcodeScannerButton#inputLiveUpdate event.
   */
  export interface BarcodeScannerButton$InputLiveUpdateEventParameters {
    /**
     * The new value of the input field.
     */
    newValue?: string;
  }

  /**
   * Event object of the BarcodeScannerButton#inputLiveUpdate event.
   */
  export type BarcodeScannerButton$InputLiveUpdateEvent = Event<
    BarcodeScannerButton$InputLiveUpdateEventParameters,
    BarcodeScannerButton
  >;

  /**
   * Parameters of the BarcodeScannerButton#scanFail event.
   */
  export interface BarcodeScannerButton$ScanFailEventParameters {}

  /**
   * Event object of the BarcodeScannerButton#scanFail event.
   */
  export type BarcodeScannerButton$ScanFailEvent = Event<
    BarcodeScannerButton$ScanFailEventParameters,
    BarcodeScannerButton
  >;

  /**
   * Parameters of the BarcodeScannerButton#scanSuccess event.
   */
  export interface BarcodeScannerButton$ScanSuccessEventParameters {
    /**
     * The the text representation of the barcode data.
     */
    text?: string;

    /**
     * The type of the barcode detected.
     */
    format?: string;

    /**
     * Indicates whether or not the user cancelled the scan.
     */
    cancelled?: boolean;
  }

  /**
   * Event object of the BarcodeScannerButton#scanSuccess event.
   */
  export type BarcodeScannerButton$ScanSuccessEvent = Event<
    BarcodeScannerButton$ScanSuccessEventParameters,
    BarcodeScannerButton
  >;
}

declare namespace sap {
  interface IUI5DefineDependencyNames {
    "sap/ndc/BarcodeScanner": undefined;

    "sap/ndc/BarcodeScannerButton": undefined;

    "sap/ndc/library": undefined;
  }
}
