import { deprecatedLibraries, deprecatedThemeLibraries } from "../../utils/deprecations.js";
import { fromYaml, getPosition } from "data-with-position";
import { MESSAGE } from "../messages.js";
export default class YamlLinter {
    #content;
    #resourcePath;
    #context;
    constructor(resourcePath, content, context) {
        this.#content = content;
        this.#resourcePath = resourcePath;
        this.#context = context;
    }
    // eslint-disable-next-line @typescript-eslint/require-await
    async lint() {
        try {
            // Split Yaml file into part documents by '---' separator
            const allDocuments = this.#content.split(/(?:\r\n|\r|\n)---/g);
            // Calculate the starting line number of each part document
            let lineNumberOffset = 0;
            allDocuments.forEach((document) => {
                // Parse content only of the current part
                const parsedYamlWithPosInfo = this.#parseYaml(document);
                // Analyze part content with line number offset
                this.#analyzeYaml(parsedYamlWithPosInfo, lineNumberOffset);
                // Update line number offset for next part
                lineNumberOffset += document.split(/\r\n|\r|\n/g).length;
            });
        }
        catch (err) {
            const message = err instanceof Error ? err.message : String(err);
            this.#context.addLintingMessage(this.#resourcePath, MESSAGE.PARSING_ERROR, { message });
        }
    }
    #parseYaml(content) {
        // Create JS object from YAML content with position information
        return fromYaml(content);
    }
    #analyzeYaml(yaml, offset) {
        // Check for deprecated libraries
        yaml?.framework?.libraries?.forEach((lib) => {
            const libraryName = lib.name.toString();
            if (deprecatedLibraries.includes(libraryName) || deprecatedThemeLibraries.includes(libraryName)) {
                const positionInfo = getPosition(lib);
                this.#context.addLintingMessage(this.#resourcePath, MESSAGE.DEPRECATED_LIBRARY, {
                    libraryName,
                }, {
                    line: positionInfo.start.line + offset,
                    column: positionInfo.start.column,
                });
            }
        });
    }
}
//# sourceMappingURL=YamlLinter.js.map