import ts from "typescript";
import { ChangeAction } from "../../../utils/textChanges.js";
import { PositionInfo } from "../../LinterContext.js";
import CallExpressionBaseFix, { CallExpressionBaseFixParams } from "./CallExpressionBaseFix.js";
import { FixHelpers } from "./Fix.js";
import { Ui5TypeInfo } from "../Ui5TypeInfo.js";
export interface CallExpressionGeneratorFixParams<GeneratorContext extends object> extends CallExpressionBaseFixParams {
    /**
     * Validation: If set to true, the fix will only be applied if the return value of the code does not use the
     * return value of the call expression.
     */
    mustNotUseReturnValue?: boolean;
    /**
     * Modules to import. If this parameter is specified, the standalone parameters "moduleName" and
     * "preferredIdentifier" must not be provided.
    */
    moduleImports?: {
        moduleName: string;
        preferredIdentifier?: string;
    }[];
    /**
     * Names of a global variable to use in the fix (e.g. "document"). If this parameter is specified, parameter
     * "globalName" must not be provided.
     *
     * The fix will be provided with the identifier names or property access strings to use via
     * the setIdentifierForGlobal method.
     *
     * For example, if there is already a conflicting identifier within the same file,
     * the fix will be provided with an alternative like "globalThis.document"
     */
    globalNames?: string[];
    /**
     * The generator function will be used to determine the value of the replacement, affecting
     * the whole call expression
     *
     * If the return value is undefined, no change will be generated.
     *
     * @param ctx - The context object that can be used to store information between validation and generation
     * @param identifierNames - The names of the identifiers that were set for the requested modules and/or global
     * 	identifiers. If both, modules and globals are requested, the array will first contain the module identifier
     * 	names in the order they have been requested, followed by the globals.
     * @param args - The arguments of the call expression, as string representations of the source code, including
     * 	line breaks and comments.
     */
    generator: (ctx: GeneratorContext, identifierNames: string[], ...args: string[]) => string | undefined;
    /**
     * Validate the arguments of the call expression, optionally using the provided checker
     *
     * This hook may also collect information that affects the generator. For that, it can store
     * information in the provided context object, which can be retrieved later in the generator function
     */
    validateArguments?: (ctx: GeneratorContext, helpers: FixHelpers, ...args: ts.Expression[]) => boolean;
}
export default class CallExpressionGeneratorFix<GeneratorContext extends object> extends CallExpressionBaseFix {
    protected params: CallExpressionGeneratorFixParams<GeneratorContext>;
    protected generatorArgs: string[] | undefined;
    protected generatorContext: GeneratorContext;
    constructor(params: CallExpressionGeneratorFixParams<GeneratorContext>, ui5TypeInfo: Ui5TypeInfo);
    visitLinterNode(node: ts.Node, sourcePosition: PositionInfo, helpers: FixHelpers): boolean;
    visitAutofixNode(node: ts.Node, position: number, sourceFile: ts.SourceFile): boolean;
    getNewModuleDependencies(): import("./Fix.js").ModuleDependencyRequest | {
        usagePosition: number;
        moduleName: string;
        preferredIdentifier?: string;
    }[] | undefined;
    getNewGlobalAccess(): import("./Fix.js").GlobalAccessRequest | import("./Fix.js").GlobalAccessRequest[] | undefined;
    generateChanges(): {
        action: ChangeAction;
        start: number;
        end: number;
        value: string;
    } | undefined;
}
