# timer-node

[![npm](https://img.shields.io/npm/v/timer-node.svg)](https://www.npmjs.com/package/timer-node)
[![npm](https://img.shields.io/npm/dm/timer-node.svg)](https://www.npmjs.com/package/timer-node)

A **lightweight** JavaScript library for measuring elapsed time in Node.js or the browser without using `setInterval` or `setTimeout`. It also has type definitions for TypeScript.

## Installation

```bash
npm install timer-node
```

## API
### Import

```javascript
// JS
const { Timer } = require('timer-node');

// TS
import { Timer, TimerOptions, Time } from 'timer-node';
```

### Timer Class

A timestamp-based timer that can be started, paused, resumed, and stopped. It calculates durations based on system time.

#### Constructor

```ts
new Timer(options?: TimerOptions)
```

- `options`: An optional configuration object with:
  - `label?: string` – A label for this timer.
  - `startTimestamp?: number` – Start time (if you want to initialize from the past).
  - `endTimestamp?: number` – End time (if already stopped).
  - `currentStartTimestamp?: number` – Most recent resume time.
  - `pauseCount?: number` – Number of times paused.
  - `accumulatedMs?: number` – Accumulated milliseconds from previous runs.

#### Common Methods

- **start()**: Starts (or restarts) the timer.
- **pause()**: Pauses the timer (if running).
- **resume()**: Resumes the timer (if paused).
- **stop()**: Stops the timer.
- **clear()**: Clears all timer data, resetting it to an unstarted state.
- **isStarted() / isPaused() / isRunning() / isStopped()**: Check the timer state.
- **ms()**: Returns the elapsed time in milliseconds.
- **time()**: Returns an object of time fractions (`d`, `h`, `m`, `s`, `ms`).
- **pauseMs() / pauseTime()**: Similar to `ms()` / `time()`, but for paused duration.
- **pauseCount()**: Returns how many times the timer has been paused.
- **format(template?: string)**: Formats the elapsed time using placeholders (`%label`, `%d`, `%h`, `%m`, `%s`, `%ms`).
- **serialize()**: Returns a JSON string representing the timer state.

#### Static Methods

- **Timer.deserialize(serializedTimer: string): Timer**\
  Re-creates a timer from a serialized string generated by `.serialize()`.

- **Timer.benchmark(fn: () => any): Timer**\
  Measures the synchronous execution time of `fn`. Returns a stopped `Timer`.

---

### Time Object

Any method returning an object breakdown of time uses the following shape:

```ts
interface Time {
  d: number;  // days
  h: number;  // hours
  m: number;  // minutes
  s: number;  // seconds
  ms: number; // milliseconds
}
```

## Example

```js
const { Timer } = require('timer-node');

const timer = new Timer({ label: 'demo' });
timer.start();

// ... some operations ...

timer.pause();
console.log('Paused at', timer.ms(), 'ms');
timer.resume();

// ... more operations ...

timer.stop();
console.log('Total elapsed:', timer.time());
// e.g. { d: 0, h: 0, m: 1, s: 12, ms: 345 }
```

## License

This library is licensed under the MIT License. See [LICENSE](https://github.com/eyas-ranjous/timer-node/blob/master/LICENSE) for details.

