import ts from "typescript";
import CallExpressionFix from "./CallExpressionFix.js";
import AccessExpressionFix from "./AccessExpressionFix.js";
import { getModuleTypeInfo, Ui5TypeInfoKind } from "../Ui5TypeInfo.js";
import getGlobalFixInfo from "./getGlobalFixInfo.js";
import CallExpressionGeneratorFix from "./CallExpressionGeneratorFix.js";
import AccessExpressionGeneratorFix from "./AccessExpressionGeneratorFix.js";
import PropertyAssignmentFix from "./PropertyAssignmentFix.js";
import PropertyAssignmentGeneratorFix from "./PropertyAssignmentGeneratorFix.js";
import ObsoleteImportFix from "./ObsoleteImportFix.js";
const AUTOFIX_COLLECTIONS = [
    "sapUiCoreFixes",
    "jqueryFixes",
    "sapMFixes",
    "sapUiLayoutFixes",
    "sapUiCompFixes",
    "globalFixes",
];
;
export default class FixFactory {
    checker;
    ambientModuleCache;
    collections;
    constructor(checker, ambientModuleCache, collections) {
        this.checker = checker;
        this.ambientModuleCache = ambientModuleCache;
        this.collections = collections;
    }
    static async create(checker, ambientModuleCache) {
        const collections = new Map();
        for (const collectionName of AUTOFIX_COLLECTIONS) {
            const { default: filter } = await import(`./collections/${collectionName}.js`);
            const libraryName = filter.getLibraryName();
            if (!libraryName) {
                throw new Error(`Fix collection ${collectionName} must define a library name in the constructor`);
            }
            collections.set(libraryName, filter);
        }
        return new FixFactory(checker, ambientModuleCache, collections);
    }
    getFix(node, ui5TypeInfo) {
        const moduleTypeInfo = getModuleTypeInfo(ui5TypeInfo);
        let collection;
        if (moduleTypeInfo) {
            collection = moduleTypeInfo.library;
        }
        else {
            // If no module type info is available, we assume it's a global fix
            collection = "global";
        }
        const filter = this.collections.get(collection);
        if (!filter) {
            return; // No collection found for the library
        }
        const fixCallback = filter.match(ui5TypeInfo);
        if (!fixCallback) {
            return;
        }
        return fixCallback(ui5TypeInfo);
    }
    getJqueryTypeInfoForFix(node, ui5TypeInfo) {
        const filter = this.collections.get("jquery");
        // Search up the AST for a node matching with a fix
        while (!filter.match(ui5TypeInfo)) {
            if (ui5TypeInfo.kind === Ui5TypeInfoKind.Module || !ui5TypeInfo.parent) {
                // We reached the module type info, but it does not match the filter
                return;
            }
            ui5TypeInfo = ui5TypeInfo.parent;
            if (ts.isCallExpression(node)) {
                node = node.expression;
            }
            node = node.expression;
        }
        return {
            ui5TypeInfo,
            node,
        };
    }
    getGlobalFixInfo(node) {
        return getGlobalFixInfo(node, this.ambientModuleCache);
    }
}
export function accessExpressionFix(params) {
    return (ui5TypeInfo) => new AccessExpressionFix(params, ui5TypeInfo);
}
export function accessExpressionGeneratorFix(params) {
    return (ui5TypeInfo) => new AccessExpressionGeneratorFix(params, ui5TypeInfo);
}
export function callExpressionFix(params) {
    return (ui5TypeInfo) => new CallExpressionFix(params, ui5TypeInfo);
}
export function callExpressionGeneratorFix(params) {
    return (ui5TypeInfo) => new CallExpressionGeneratorFix(params, ui5TypeInfo);
}
export function propertyAssignmentFix(params) {
    return (_ui5TypeInfo) => new PropertyAssignmentFix(params);
}
export function propertyAssignmentGeneratorFix(params) {
    return (_ui5TypeInfo) => new PropertyAssignmentGeneratorFix(params);
}
export function obsoleteImportFix(params) {
    return (_ui5TypeInfo) => new ObsoleteImportFix(params);
}
//# sourceMappingURL=FixFactory.js.map