"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const execa_1 = __importDefault(require("execa"));
const globby_1 = __importDefault(require("globby"));
const cheerio_1 = __importDefault(require("cheerio"));
const install_1 = require("../lib/install");
const run_jscodeshift_1 = __importDefault(require("../lib/run-jscodeshift"));
const html_to_react_attributes_1 = __importDefault(require("../lib/html-to-react-attributes"));
const index_to_component_1 = require("../lib/cra-to-next/index-to-component");
const global_css_transform_1 = require("../lib/cra-to-next/global-css-transform");
const feedbackMessage = `Please share any feedback on the migration here: https://github.com/vercel/next.js/discussions/25858`;
// log error and exit without new stacktrace
function fatalMessage(...logs) {
    console.error(...logs, `\n${feedbackMessage}`);
    process.exit(1);
}
const craTransformsPath = path_1.default.join('../lib/cra-to-next');
const globalCssTransformPath = require.resolve(path_1.default.join(craTransformsPath, 'global-css-transform.js'));
const indexTransformPath = require.resolve(path_1.default.join(craTransformsPath, 'index-to-component.js'));
class CraTransform {
    constructor(files, flags) {
        this.isDryRun = flags.dry;
        this.jscodeShiftFlags = flags;
        this.appDir = this.validateAppDir(files);
        this.packageJsonPath = path_1.default.join(this.appDir, 'package.json');
        this.packageJsonData = this.loadPackageJson();
        this.shouldLogInfo = flags.print || flags.dry;
        this.pagesDir = this.getPagesDir();
        this.installClient = this.checkForYarn() ? 'yarn' : 'npm';
        const { dependencies, devDependencies } = this.packageJsonData;
        const hasDep = (dep) => (dependencies === null || dependencies === void 0 ? void 0 : dependencies[dep]) || (devDependencies === null || devDependencies === void 0 ? void 0 : devDependencies[dep]);
        this.isCra = hasDep('react-scripts');
        this.isVite = !this.isCra && hasDep('vite');
        if (!this.isCra && !this.isVite) {
            fatalMessage(`Error: react-scripts was not detected, is this a CRA project?`);
        }
        this.shouldUseTypeScript =
            fs_1.default.existsSync(path_1.default.join(this.appDir, 'tsconfig.json')) ||
                globby_1.default.sync('src/**/*.{ts,tsx}', {
                    cwd: path_1.default.join(this.appDir, 'src'),
                }).length > 0;
        this.indexPage = globby_1.default.sync([`${this.isCra ? 'index' : 'main'}.{js,jsx,ts,tsx}`], {
            cwd: path_1.default.join(this.appDir, 'src'),
        })[0];
        if (!this.indexPage) {
            fatalMessage('Error: unable to find `src/index`');
        }
    }
    transform() {
        return __awaiter(this, void 0, void 0, function* () {
            console.log('Transforming CRA project at:', this.appDir);
            // convert src/index.js to a react component to render
            // inside of Next.js instead of the custom render root
            const indexTransformRes = yield (0, run_jscodeshift_1.default)(indexTransformPath, Object.assign(Object.assign({}, this.jscodeShiftFlags), { silent: true, verbose: 0 }), [path_1.default.join(this.appDir, 'src', this.indexPage)]);
            if (indexTransformRes.error > 0) {
                fatalMessage(`Error: failed to apply transforms for src/${this.indexPage}, please check for syntax errors to continue`);
            }
            if (index_to_component_1.indexContext.multipleRenderRoots) {
                fatalMessage(`Error: multiple ReactDOM.render roots in src/${this.indexPage}, migrate additional render roots to use portals instead to continue.\n` +
                    `See here for more info: https://react.dev/reference/react-dom/createPortal`);
            }
            if (index_to_component_1.indexContext.nestedRender) {
                fatalMessage(`Error: nested ReactDOM.render found in src/${this.indexPage}, please migrate this to a top-level render (no wrapping functions) to continue`);
            }
            // comment out global style imports and collect them
            // so that we can add them to _app
            const globalCssRes = yield (0, run_jscodeshift_1.default)(globalCssTransformPath, Object.assign({}, this.jscodeShiftFlags), [this.appDir]);
            if (globalCssRes.error > 0) {
                fatalMessage(`Error: failed to apply transforms for src/${this.indexPage}, please check for syntax errors to continue`);
            }
            if (!this.isDryRun) {
                yield fs_1.default.promises.mkdir(path_1.default.join(this.appDir, this.pagesDir));
            }
            this.logCreate(this.pagesDir);
            if (global_css_transform_1.globalCssContext.reactSvgImports.size > 0) {
                // This de-opts webpack 5 since svg/webpack doesn't support webpack 5 yet,
                // so we don't support this automatically
                fatalMessage(`Error: import {ReactComponent} from './logo.svg' is not supported, please use normal SVG imports to continue.\n` +
                    `React SVG imports found in:\n${[
                        ...global_css_transform_1.globalCssContext.reactSvgImports,
                    ].join('\n')}`);
            }
            yield this.updatePackageJson();
            yield this.createNextConfig();
            yield this.updateGitIgnore();
            yield this.createPages();
        });
    }
    checkForYarn() {
        try {
            const userAgent = process.env.npm_config_user_agent;
            if (userAgent) {
                return Boolean(userAgent && userAgent.startsWith('yarn'));
            }
            execa_1.default.sync('yarnpkg', ['--version'], { stdio: 'ignore' });
            return true;
        }
        catch (e) {
            console.log('error', e);
            return false;
        }
    }
    logCreate(...args) {
        if (this.shouldLogInfo) {
            console.log('Created:', ...args);
        }
    }
    logModify(...args) {
        if (this.shouldLogInfo) {
            console.log('Modified:', ...args);
        }
    }
    logInfo(...args) {
        if (this.shouldLogInfo) {
            console.log(...args);
        }
    }
    createPages() {
        return __awaiter(this, void 0, void 0, function* () {
            // load public/index.html and add tags to _document
            const htmlContent = yield fs_1.default.promises.readFile(path_1.default.join(this.appDir, `${this.isCra ? 'public/' : ''}index.html`), 'utf8');
            const $ = cheerio_1.default.load(htmlContent);
            // note: title tag and meta[viewport] needs to be placed in _app
            // not _document
            const titleTag = $('title')[0];
            const metaViewport = $('meta[name="viewport"]')[0];
            const headTags = $('head').children();
            const bodyTags = $('body').children();
            const pageExt = this.shouldUseTypeScript ? 'tsx' : 'js';
            const appPage = path_1.default.join(this.pagesDir, `_app.${pageExt}`);
            const documentPage = path_1.default.join(this.pagesDir, `_document.${pageExt}`);
            const catchAllPage = path_1.default.join(this.pagesDir, `[[...slug]].${pageExt}`);
            const gatherTextChildren = (children) => {
                return children
                    .map((child) => {
                    if (child.type === 'text') {
                        return child.data;
                    }
                    return '';
                })
                    .join('');
            };
            const serializeAttrs = (attrs) => {
                const attrStr = Object.keys(attrs || {})
                    .map((name) => {
                    const reactName = html_to_react_attributes_1.default[name] || name;
                    const value = attrs[name];
                    // allow process.env access to work dynamically still
                    if (value.match(/%([a-zA-Z0-9_]{0,})%/)) {
                        return `${reactName}={\`${value.replace(/%([a-zA-Z0-9_]{0,})%/g, (subStr) => {
                            return `\${process.env.${subStr.slice(1, -1)}}`;
                        })}\`}`;
                    }
                    return `${reactName}="${value}"`;
                })
                    .join(' ');
                return attrStr.length > 0 ? ` ${attrStr}` : '';
            };
            const serializedHeadTags = [];
            const serializedBodyTags = [];
            headTags.map((_index, element) => {
                if (element.tagName === 'title' ||
                    (element.tagName === 'meta' && element.attribs.name === 'viewport')) {
                    return element;
                }
                let hasChildren = element.children.length > 0;
                let serializedAttrs = serializeAttrs(element.attribs);
                if (element.tagName === 'script' || element.tagName === 'style') {
                    hasChildren = false;
                    serializedAttrs += ` dangerouslySetInnerHTML={{ __html: \`${gatherTextChildren(element.children).replace(/`/g, '\\`')}\` }}`;
                }
                serializedHeadTags.push(hasChildren
                    ? `<${element.tagName}${serializedAttrs}>${gatherTextChildren(element.children)}</${element.tagName}>`
                    : `<${element.tagName}${serializedAttrs} />`);
                return element;
            });
            bodyTags.map((_index, element) => {
                if (element.tagName === 'div' && element.attribs.id === 'root') {
                    return element;
                }
                let hasChildren = element.children.length > 0;
                let serializedAttrs = serializeAttrs(element.attribs);
                if (element.tagName === 'script' || element.tagName === 'style') {
                    hasChildren = false;
                    serializedAttrs += ` dangerouslySetInnerHTML={{ __html: \`${gatherTextChildren(element.children).replace(/`/g, '\\`')}\` }}`;
                }
                serializedHeadTags.push(hasChildren
                    ? `<${element.tagName}${serializedAttrs}>${gatherTextChildren(element.children)}</${element.tagName}>`
                    : `<${element.tagName}${serializedAttrs} />`);
                return element;
            });
            if (!this.isDryRun) {
                yield fs_1.default.promises.writeFile(path_1.default.join(this.appDir, appPage), `${global_css_transform_1.globalCssContext.cssImports.size === 0
                    ? ''
                    : [...global_css_transform_1.globalCssContext.cssImports]
                        .map((file) => {
                        if (!this.isCra) {
                            file = file.startsWith('/') ? file.slice(1) : file;
                        }
                        return `import '${file.startsWith('/')
                            ? path_1.default.relative(path_1.default.join(this.appDir, this.pagesDir), file)
                            : file}'`;
                    })
                        .join('\n') + '\n'}${titleTag ? `import Head from 'next/head'` : ''}

export default function MyApp({ Component, pageProps}) {
  ${titleTag || metaViewport
                    ? `return (
    <>
      <Head>
        ${titleTag
                        ? `<title${serializeAttrs(titleTag.attribs)}>${gatherTextChildren(titleTag.children)}</title>`
                        : ''}
        ${metaViewport ? `<meta${serializeAttrs(metaViewport.attribs)} />` : ''}
      </Head>
      
      <Component {...pageProps} />
    </>
  )`
                    : 'return <Component {...pageProps} />'}
}
`);
                yield fs_1.default.promises.writeFile(path_1.default.join(this.appDir, documentPage), `import Document, { Html, Head, Main, NextScript } from 'next/document'

class MyDocument extends Document {
  render() {
    return (
      <Html${serializeAttrs($('html').attr())}>
        <Head>
          ${serializedHeadTags.join('\n          ')}
        </Head>
        
        <body${serializeAttrs($('body').attr())}>
          <Main />
          <NextScript />
          ${serializedBodyTags.join('\n          ')}
        </body>
      </Html>
    )
  }
}

export default MyDocument      
`);
                const relativeIndexPath = path_1.default.relative(path_1.default.join(this.appDir, this.pagesDir), path_1.default.join(this.appDir, 'src', this.isCra ? '' : 'main'));
                // TODO: should we default to ssr: true below and recommend they
                // set it to false if they encounter errors or prefer the more safe
                // option to prevent their first start from having any errors?
                yield fs_1.default.promises.writeFile(path_1.default.join(this.appDir, catchAllPage), `// import NextIndexWrapper from '${relativeIndexPath}'

// next/dynamic is used to prevent breaking incompatibilities 
// with SSR from window.SOME_VAR usage, if this is not used
// next/dynamic can be removed to take advantage of SSR/prerendering
import dynamic from 'next/dynamic'

// try changing "ssr" to true below to test for incompatibilities, if
// no errors occur the above static import can be used instead and the
// below removed
const NextIndexWrapper = dynamic(() => import('${relativeIndexPath}'), { ssr: false })

export default function Page(props) {
  return <NextIndexWrapper {...props} />
}
`);
            }
            this.logCreate(appPage);
            this.logCreate(documentPage);
            this.logCreate(catchAllPage);
        });
    }
    updatePackageJson() {
        return __awaiter(this, void 0, void 0, function* () {
            // rename react-scripts -> next and react-scripts test -> jest
            // add needed dependencies for webpack compatibility
            const newDependencies = [
                // TODO: do we want to install jest automatically?
                {
                    name: 'next',
                    version: 'latest',
                },
            ];
            const packageName = this.isCra ? 'react-scripts' : 'vite';
            const packagesToRemove = {
                [packageName]: undefined,
            };
            const neededDependencies = [];
            const { devDependencies, dependencies, scripts } = this.packageJsonData;
            for (const dep of newDependencies) {
                if (!(devDependencies === null || devDependencies === void 0 ? void 0 : devDependencies[dep.name]) && !(dependencies === null || dependencies === void 0 ? void 0 : dependencies[dep.name])) {
                    neededDependencies.push(`${dep.name}@${dep.version}`);
                }
            }
            this.logInfo(`Installing ${neededDependencies.join(' ')} with ${this.installClient}`);
            if (!this.isDryRun) {
                yield fs_1.default.promises.writeFile(this.packageJsonPath, JSON.stringify(Object.assign(Object.assign({}, this.packageJsonData), { scripts: Object.keys(scripts).reduce((prev, cur) => {
                        const command = scripts[cur];
                        prev[cur] = command;
                        if (command === packageName) {
                            prev[cur] = 'next dev';
                        }
                        if (command.includes(`${packageName} `)) {
                            prev[cur] = command.replace(`${packageName} `, command.includes(`${packageName} test`) ? 'jest ' : 'next ');
                        }
                        if (cur === 'eject') {
                            prev[cur] = undefined;
                        }
                        // TODO: do we want to map start -> next start instead of CRA's
                        // default of mapping starting to dev mode?
                        if (cur === 'start') {
                            prev[cur] = prev[cur].replace('next start', 'next dev');
                            prev['start-production'] = 'next start';
                        }
                        return prev;
                    }, {}), dependencies: Object.assign(Object.assign({}, dependencies), packagesToRemove), devDependencies: Object.assign(Object.assign({}, devDependencies), packagesToRemove) }), null, 2));
                yield (0, install_1.install)(this.appDir, neededDependencies, {
                    useYarn: this.installClient === 'yarn',
                    // do we want to detect offline as well? they might not
                    // have next in the local cache already
                    isOnline: true,
                });
            }
        });
    }
    updateGitIgnore() {
        return __awaiter(this, void 0, void 0, function* () {
            // add Next.js specific items to .gitignore e.g. '.next'
            const gitignorePath = path_1.default.join(this.appDir, '.gitignore');
            let ignoreContent = yield fs_1.default.promises.readFile(gitignorePath, 'utf8');
            const nextIgnores = (yield fs_1.default.promises.readFile(path_1.default.join(path_1.default.dirname(globalCssTransformPath), 'gitignore'), 'utf8')).split('\n');
            if (!this.isDryRun) {
                for (const ignore of nextIgnores) {
                    if (!ignoreContent.includes(ignore)) {
                        ignoreContent += `\n${ignore}`;
                    }
                }
                yield fs_1.default.promises.writeFile(gitignorePath, ignoreContent);
            }
            this.logModify('.gitignore');
        });
    }
    createNextConfig() {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.isDryRun) {
                const { proxy, homepage } = this.packageJsonData;
                const homepagePath = new URL(homepage || '/', 'http://example.com')
                    .pathname;
                yield fs_1.default.promises.writeFile(path_1.default.join(this.appDir, 'next.config.js'), `module.exports = {${proxy
                    ? `
  async rewrites() {
    return {
      fallback: [
        {
          source: '/:path*',
          destination: '${proxy}'
        }
      ]
    }
  },`
                    : ''}
  env: {
    PUBLIC_URL: '${homepagePath === '/' ? '' : homepagePath || ''}'
  },
  experimental: {
    craCompat: true,
  },
  // Remove this to leverage Next.js' static image handling
  // read more here: https://nextjs.org/docs/api-reference/next/image
  images: {
    disableStaticImages: true
  }  
}
`);
            }
            this.logCreate('next.config.js');
        });
    }
    getPagesDir() {
        // prefer src/pages as CRA uses the src dir by default
        // and attempt falling back to top-level pages dir
        let pagesDir = 'src/pages';
        if (fs_1.default.existsSync(path_1.default.join(this.appDir, pagesDir))) {
            pagesDir = 'pages';
        }
        if (fs_1.default.existsSync(path_1.default.join(this.appDir, pagesDir))) {
            fatalMessage(`Error: a "./pages" directory already exists, please rename to continue`);
        }
        return pagesDir;
    }
    loadPackageJson() {
        let packageJsonData;
        try {
            packageJsonData = JSON.parse(fs_1.default.readFileSync(this.packageJsonPath, 'utf8'));
        }
        catch (err) {
            fatalMessage(`Error: failed to load package.json from ${this.packageJsonPath}, ensure provided directory is root of CRA project`);
        }
        return packageJsonData;
    }
    validateAppDir(files) {
        if (files.length > 1) {
            fatalMessage(`Error: only one directory should be provided for the cra-to-next transform, received ${files.join(', ')}`);
        }
        const appDir = path_1.default.join(process.cwd(), files[0]);
        let isValidDirectory = false;
        try {
            isValidDirectory = fs_1.default.lstatSync(appDir).isDirectory();
        }
        catch (err) {
            // not a valid directory
        }
        if (!isValidDirectory) {
            fatalMessage(`Error: invalid directory provided for the cra-to-next transform, received ${appDir}`);
        }
        return appDir;
    }
}
function transformer(files, flags) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const craTransform = new CraTransform(files, flags);
            yield craTransform.transform();
            console.log(`CRA to Next.js migration complete`, `\n${feedbackMessage}`);
        }
        catch (err) {
            fatalMessage(`Error: failed to complete transform`, err);
        }
    });
}
exports.default = transformer;
//# sourceMappingURL=cra-to-next.js.map