"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Options = void 0;
const parser_1 = require("@codemod/parser");
const fs_1 = require("fs");
const path_1 = require("path");
const resolve_1 = require("resolve");
const Config_1 = require("./Config");
const extensions_1 = require("./extensions");
class Options {
    constructor(args) {
        this.args = args;
    }
    parse() {
        const config = new Config_1.ConfigBuilder();
        let lastPlugin;
        for (let i = 0; i < this.args.length; i++) {
            const arg = this.args[i];
            switch (arg) {
                case '-p':
                case '--plugin':
                    i++;
                    lastPlugin = this.args[i];
                    config.addLocalPlugin(lastPlugin);
                    break;
                case '--remote-plugin':
                    i++;
                    lastPlugin = this.args[i];
                    config.addRemotePlugin(lastPlugin);
                    break;
                case '-o':
                case '--plugin-options': {
                    i++;
                    const value = this.args[i];
                    let name;
                    let optionsRaw;
                    if (value.startsWith('@')) {
                        if (!lastPlugin) {
                            throw new Error(`${arg} must follow --plugin or --remote-plugin if no name is given`);
                        }
                        optionsRaw = (0, fs_1.readFileSync)(value.slice(1), 'utf8');
                        name = lastPlugin;
                    }
                    else if (/^\s*{/.test(value)) {
                        if (!lastPlugin) {
                            throw new Error(`${arg} must follow --plugin or --remote-plugin if no name is given`);
                        }
                        optionsRaw = value;
                        name = lastPlugin;
                    }
                    else {
                        const nameAndOptions = value.split('=');
                        name = nameAndOptions[0];
                        optionsRaw = nameAndOptions[1];
                        if (optionsRaw.startsWith('@')) {
                            optionsRaw = (0, fs_1.readFileSync)(optionsRaw.slice(1), 'utf8');
                        }
                    }
                    try {
                        config.setOptionsForPlugin(JSON.parse(optionsRaw), name);
                    }
                    catch (err) {
                        throw new Error(`unable to parse JSON config for ${name}: ${optionsRaw}`);
                    }
                    break;
                }
                case '--parser-plugins': {
                    i++;
                    const value = this.args[i];
                    if (!value) {
                        throw new Error(`${arg} must be followed by a comma-separated list`);
                    }
                    for (const plugin of value.split(',')) {
                        if ((0, parser_1.isParserPluginName)(plugin)) {
                            config.addParserPlugin(plugin);
                        }
                        else {
                            throw new Error(`unknown parser plugin: ${plugin}`);
                        }
                    }
                    break;
                }
                case '-r':
                case '--require':
                    i++;
                    config.addRequire(getRequirableModulePath(this.args[i]));
                    break;
                case '--transpile-plugins':
                case '--no-transpile-plugins':
                    config.transpilePlugins(arg === '--transpile-plugins');
                    break;
                case '--extensions':
                    i++;
                    config.extensions(new Set(this.args[i]
                        .split(',')
                        .map((ext) => (ext[0] === '.' ? ext : `.${ext}`))));
                    break;
                case '--add-extension':
                    i++;
                    config.addExtension(this.args[i]);
                    break;
                case '--source-type': {
                    i++;
                    const sourceType = this.args[i];
                    if (sourceType === 'module' ||
                        sourceType === 'script' ||
                        sourceType === 'unambiguous') {
                        config.sourceType(sourceType);
                    }
                    else {
                        throw new Error(`expected '--source-type' to be one of "module", "script", ` +
                            `or "unambiguous" but got: "${sourceType}"`);
                    }
                    break;
                }
                case '-s':
                case '--stdio':
                    config.stdio(true);
                    break;
                case '-h':
                case '--help':
                    return { kind: 'help' };
                case '--version':
                    return { kind: 'version' };
                case '-d':
                case '--dry':
                    config.dry(true);
                    break;
                default:
                    if (arg[0] === '-') {
                        throw new Error(`unexpected option: ${arg}`);
                    }
                    else {
                        config.addSourcePath(arg);
                    }
                    break;
            }
        }
        return {
            kind: 'run',
            config: config.build(),
        };
    }
}
exports.Options = Options;
/**
 * Gets a path that can be passed to `require` for a given module path.
 */
function getRequirableModulePath(modulePath) {
    if ((0, fs_1.existsSync)(modulePath)) {
        return (0, path_1.resolve)(modulePath);
    }
    for (const ext of extensions_1.RequireableExtensions) {
        if ((0, fs_1.existsSync)(modulePath + ext)) {
            return (0, path_1.resolve)(modulePath + ext);
        }
    }
    return (0, resolve_1.sync)(modulePath, { basedir: process.cwd() });
}
//# sourceMappingURL=Options.js.map