/*
 * Copyright 2021 the original author or authors.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * https://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.openrewrite.java.security;

import org.openrewrite.*;
import org.openrewrite.internal.lang.Nullable;
import org.openrewrite.java.security.xml.DocumentBuilderFactoryFixVisitor;
import org.openrewrite.java.security.xml.ExternalDTDAccumulator;
import org.openrewrite.java.security.xml.TransformerFactoryFixVisitor;
import org.openrewrite.java.security.xml.XmlInputFactoryFixVisitor;

import java.util.Arrays;
import java.util.List;

public class XmlParserXXEVulnerability extends ScanningRecipe<ExternalDTDAccumulator> {

    @Override
    public String getDisplayName() {
        return "XML parser XXE vulnerability";
    }

    @Override
    public String getDescription() {
        return "Avoid exposing dangerous features of the XML parser by updating certain factory settings.";
    }

    @Override
    public ExternalDTDAccumulator getInitialValue(ExecutionContext ctx) {
        return ExternalDTDAccumulator.create();
    }

    @Override
    public TreeVisitor<?, ExecutionContext> getScanner(ExternalDTDAccumulator acc) {
        return acc.scanner();
    }

    @Override
    public TreeVisitor<?, ExecutionContext> getVisitor(ExternalDTDAccumulator acc) {
        List<TreeVisitor<?, ExecutionContext>> xmlVisitors = Arrays.asList(
                XmlInputFactoryFixVisitor.create(acc),
                TransformerFactoryFixVisitor.create(acc),
                DocumentBuilderFactoryFixVisitor.create(acc)
        );
        return new TreeVisitor<Tree, ExecutionContext>() {
            @Override
            public Tree visit(@Nullable Tree tree, ExecutionContext executionContext) {
                for (TreeVisitor<?, ExecutionContext> xmlVisitor : xmlVisitors) {
                    //noinspection DataFlowIssue
                    if (xmlVisitor.isAcceptable((SourceFile) tree, executionContext)) {
                        tree = xmlVisitor.visit(tree, executionContext);
                    }
                }
                return tree;
            }
        };
    }
}
