/*
 * Copyright 2021 the original author or authors.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * https://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.openrewrite.nodejs;

import com.fasterxml.jackson.databind.MappingIterator;
import com.fasterxml.jackson.dataformat.csv.CsvMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import lombok.EqualsAndHashCode;
import lombok.Value;
import org.openrewrite.*;
import org.openrewrite.internal.StringUtils;
import org.openrewrite.internal.lang.Nullable;
import org.openrewrite.json.JsonIsoVisitor;
import org.openrewrite.json.JsonPathMatcher;
import org.openrewrite.json.tree.Json;
import org.openrewrite.marker.SearchResult;
import org.openrewrite.nodejs.internal.StaticVersionComparator;
import org.openrewrite.nodejs.internal.Version;
import org.openrewrite.nodejs.internal.VersionParser;
import org.openrewrite.nodejs.search.IsPackageJson;
import org.openrewrite.nodejs.search.IsPackageLockJson;
import org.openrewrite.nodejs.table.VulnerabilityReport;
import org.openrewrite.semver.LatestPatch;

import java.io.IOException;
import java.io.InputStream;
import java.util.*;
import java.util.stream.Collectors;

import static java.util.stream.Collectors.joining;

@Value
@EqualsAndHashCode(callSuper = false)
public class DependencyVulnerabilityCheck extends ScanningRecipe<DependencyVulnerabilityCheck.Accumulator> {
    transient VersionParser versionParser = new VersionParser();
    transient VulnerabilityReport report = new VulnerabilityReport(this);

    @Option(displayName = "Add search markers",
            description = "Report each vulnerability as search result markers. " +
                          "When enabled you can see which dependencies are bringing in vulnerable transitives in the diff view. " +
                          "By default these markers are omitted, making it easier to see version upgrades within the diff.",
            required = false)
    @Nullable
    Boolean addMarkers;

    @Override
    public String getDisplayName() {
        return "Find and fix vulnerable npm dependencies";
    }

    @Override
    public String getDescription() {
        //language=markdown
        return "This software composition analysis (SCA) tool detects and upgrades dependencies with publicly disclosed vulnerabilities. " +
               "This recipe both generates a report of vulnerable dependencies and upgrades to newer versions with fixes. " +
               "This recipe **only** upgrades to the latest **patch** version.  If a minor or major upgrade is required to reach the fixed version, this recipe will not make any changes. " +
               "Vulnerability information comes from the [GitHub Security Advisory Database](https://docs.github.com/en/code-security/security-advisories/global-security-advisories/about-the-github-advisory-database), " +
               "which aggregates vulnerability data from several public databases, including the [National Vulnerability Database](https://nvd.nist.gov/) maintained by the United States government. " +
               "Dependencies following [Semantic Versioning](https://semver.org/) will see their _patch_ version updated where applicable.";
    }

    @Value
    public static class Accumulator {
        Map<String, List<Vulnerability>> db;
        Map<NameVersion, Set<Vulnerability>> vulnerabilities;

        @Value
        static class NameVersion {
            /**
             * The name of the package as specified in the package.json.
             */
            String name;

            /**
             * The resolved version actually in use, which may be different from the version specified in the package.json.
             */
            String version;
        }
    }

    @Override
    public Accumulator getInitialValue(ExecutionContext ctx) {
        CsvMapper csvMapper = new CsvMapper();
        csvMapper.registerModule(new JavaTimeModule());
        Map<String, List<Vulnerability>> db = new HashMap<>();

        try (InputStream resourceAsStream = DependencyVulnerabilityCheck.class.getResourceAsStream("/advisories-npm.csv");
             MappingIterator<Vulnerability> vs = csvMapper.readerWithSchemaFor(Vulnerability.class).readValues(resourceAsStream)) {
            while (vs.hasNextValue()) {
                Vulnerability v = vs.nextValue();
                db.computeIfAbsent(v.getPackageName(), g -> new ArrayList<>()).add(v);
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return new Accumulator(db, new HashMap<>());
    }

    @Override
    public TreeVisitor<?, ExecutionContext> getScanner(Accumulator acc) {
        return Preconditions.check(new IsPackageLockJson<>(), new JsonIsoVisitor<ExecutionContext>() {
            @Override
            public Json.Document visitDocument(Json.Document document, ExecutionContext ctx) {
                NodeResolutionResult nodeResolutionResult = NodeResolutionResult.fromPackageLockJson(document);
                // Find all vulnerable dependencies and add them to the accumulator
                findVulnerabilities(nodeResolutionResult.getDependencies());
                findVulnerabilities(nodeResolutionResult.getDevDependencies());
                return document;
            }

            private void findVulnerabilities(Collection<Dependency> dependencies) {
                for (Dependency dependency : dependencies) {
                    for (Vulnerability v : acc.getDb().getOrDefault(dependency.getName(), Collections.emptyList())) {
                        String resolvedVersion = dependency.getResolved() == null ? null : dependency.getResolved().getVersion();
                        acc.getVulnerabilities()
                                .computeIfAbsent(new Accumulator.NameVersion(dependency.getName(), resolvedVersion), nv -> new LinkedHashSet<>())
                                .add(v);
                    }
                }
            }
        });
    }

    @Override
    public Collection<SourceFile> generate(Accumulator acc, ExecutionContext ctx) {
        Comparator<Version> vc = new StaticVersionComparator();
        LatestPatch latestPatch = new LatestPatch(null);
        for (Map.Entry<Accumulator.NameVersion, Set<Vulnerability>> vulnerabilitiesByPackage : acc.getVulnerabilities().entrySet()) {
            Accumulator.NameVersion nameVersion = vulnerabilitiesByPackage.getKey();
            Version resolvedVersion = versionParser.transform(nameVersion.getVersion());
            for (Vulnerability v : vulnerabilitiesByPackage.getValue()) {
                if (vc.compare(resolvedVersion, versionParser.transform(v.getFixedVersion())) < 0) {
                    boolean fixWithPatchVersionUpdateOnly = latestPatch.isValid(nameVersion.getVersion(), v.getFixedVersion()) &&
                                                            latestPatch.compare(nameVersion.getVersion(), nameVersion.getVersion(), v.getFixedVersion()) < 0;
                    // Insert a row into the report for each vulnerability
                    report.insertRow(ctx, new VulnerabilityReport.Row(
                            v.getCve(),
                            nameVersion.getName(),
                            nameVersion.getVersion(),
                            v.getFixedVersion(),
                            fixWithPatchVersionUpdateOnly,
                            v.getSummary(),
                            v.getSeverity().toString(),
                            0,
                            v.getCwes()
                    ));
                }
            }
        }
        return Collections.emptyList();
    }

    @Override
    public TreeVisitor<?, ExecutionContext> getVisitor(Accumulator acc) {
        JsonPathMatcher dependency = new JsonPathMatcher("$.dependencies");
        JsonPathMatcher devDependencies = new JsonPathMatcher("$.devDependencies");
        Comparator<Version> vc = new StaticVersionComparator();
        LatestPatch latestPatch = new LatestPatch(null);
        return Preconditions.check(new IsPackageJson<>(), new JsonIsoVisitor<ExecutionContext>() {
            @Override
            public Json.Document visitDocument(Json.Document document, ExecutionContext ctx) {
                Json.Document d = super.visitDocument(document, ctx);
                for (Map.Entry<Accumulator.NameVersion, Set<Vulnerability>> entry : acc.getVulnerabilities().entrySet()) {
                    String resolvedVersion = entry.getKey().getVersion();
                    for (Vulnerability v : entry.getValue()) {
                        boolean fixWithPatchVersionUpdateOnly = latestPatch.isValid(resolvedVersion, v.getFixedVersion()) &&
                                                                latestPatch.compare(resolvedVersion, resolvedVersion, v.getFixedVersion()) < 0;
                        if (fixWithPatchVersionUpdateOnly) {
                            d = (Json.Document) new UpgradeDependencyVersion(v.getPackageName(), '^' + v.getFixedVersion())
                                    .getVisitor()
                                    .visitNonNull(d, ctx, getCursor().getParentOrThrow());
                        }
                    }
                }
                return d;
            }

            @Override
            public Json.Member visitMember(Json.Member member, ExecutionContext ctx) {
                Json.Member m = super.visitMember(member, ctx);
                if (!Boolean.TRUE.equals(addMarkers)) {
                    return m;
                }

                Cursor maybeDependencies = getCursor().getParent(2);
                if (maybeDependencies != null && (dependency.matches(maybeDependencies) || devDependencies.matches(maybeDependencies))) {
                    String name = ((Json.Literal) member.getKey()).getValue().toString();
                    for (Map.Entry<Accumulator.NameVersion, Set<Vulnerability>> entry : acc.getVulnerabilities().entrySet()) {
                        Accumulator.NameVersion nameVersion = entry.getKey();
                        if (nameVersion.getName().equals(name)) {
                            Version resolvedVersion = versionParser.transform(nameVersion.getVersion());
                            List<Vulnerability> applicableVulnerabilities = entry.getValue().stream()
                                    .filter(v -> StringUtils.isBlank(v.getFixedVersion()) ||
                                                 vc.compare(resolvedVersion, versionParser.transform(v.getFixedVersion())) < 0)
                                    .collect(Collectors.toList());
                            if (!applicableVulnerabilities.isEmpty()) {
                                return SearchResult.found(m,
                                        "This dependency has the following vulnerabilities:\n" +
                                        applicableVulnerabilities.stream()
                                                .map(v -> String.format("%s (%s severity%s) - %s",
                                                        v.getCve(),
                                                        v.getSeverity(),
                                                        StringUtils.isBlank(v.getFixedVersion()) ? "" : ", fixed in " + v.getFixedVersion(),
                                                        v.getSummary()))
                                                .collect(joining("\n")));
                            }
                        }
                    }
                }

                return m;
            }
        });
    }
}
